﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mem.h>

#include "htclow_Allocator.h"

namespace nn { namespace htclow { namespace server {

namespace {
    const size_t DefaultHeapSize = 8 * 1024 * 1024;

    // 16KB (管理領域) + 4KB (スタックサイズ) * 3 (スレッド数) + 4KB (予備)
    // TODO: mem::StandardAllocator が最低限要求する管理領域が 16KB もあって非効率なので、アロケータを自作する
    const size_t ThreadStackHeapSize = 32 * 1024;

    NN_ALIGNAS(4096) uint8_t g_DefaultHeap[DefaultHeapSize];
    NN_ALIGNAS(4096) uint8_t g_ThreadStackHeap[ThreadStackHeapSize];

    nn::mem::StandardAllocator g_DefaultAllocator;
    nn::mem::StandardAllocator g_ThreadStackAllocator;

#if !defined(NN_SDK_BUILD_RELEASE)
    bool g_Initialized = false;
#endif
}

void InitializeAllocator() NN_NOEXCEPT
{
    g_DefaultAllocator.Initialize(g_DefaultHeap, DefaultHeapSize);
    g_ThreadStackAllocator.Initialize(g_ThreadStackHeap, ThreadStackHeapSize);

#if !defined(NN_SDK_BUILD_RELEASE)
    g_Initialized = true;
#endif
}

void* AllocateDefault(size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    return g_DefaultAllocator.Allocate(size);
}

void* AllocateThreadStack(size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    return g_ThreadStackAllocator.Allocate(size, nn::os::ThreadStackAlignment);
}

void FreeDefault(void* p) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    return g_DefaultAllocator.Free(p);
}

void FreeThreadStack(void* p) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    return g_ThreadStackAllocator.Free(p);
}

size_t GetDefaultAllocatorUsage() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    return g_DefaultAllocator.Hash().allocSize;
}

void PrintDefaultAllocatorUsage() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Initialized);
    g_DefaultAllocator.Dump();
}

}}}
