﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/htclow.h>

#include <nn/os/os_SdkMutex.h>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/sf/sf_Types.h>

#include "htcfs_HeaderFactory.h"

namespace nn { namespace htcfs { namespace server {

// htcfs サーバライブラリの主要な機能を実装するシングルトン
class HtcfsImpl
{
    NN_DISALLOW_COPY(HtcfsImpl);
    NN_DISALLOW_MOVE(HtcfsImpl);

public:
    static HtcfsImpl* GetInstance() NN_NOEXCEPT;

    nn::Result GetEntryType(nn::fs::DirectoryEntryType* pOutType, const char* pPath) NN_NOEXCEPT;
    nn::Result OpenFile(int32_t* pOutHandle, const char* pPath, nn::fs::OpenMode mode) NN_NOEXCEPT;
    nn::Result CloseFile(int32_t handle) NN_NOEXCEPT;
    nn::Result GetPriorityForFile(int32_t* pOutPriority, int32_t handle) NN_NOEXCEPT;
    nn::Result SetPriorityForFile(int32_t priority, int32_t handle) NN_NOEXCEPT;
    nn::Result CreateFile(const char* pPath, int64_t size) NN_NOEXCEPT;
    nn::Result DeleteFile(const char* pPath) NN_NOEXCEPT;
    nn::Result RenameFile(const char* pFromName, const char* pToName) NN_NOEXCEPT;
    nn::Result FileExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT;
    nn::Result ReadFile(int64_t* pOutSize, void* pOutBuffer, int32_t handle, int64_t offset, int64_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;
    nn::Result WriteFile(const void* pBuffer, int32_t handle, int64_t offset, int64_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT;
    nn::Result FlushFile(int32_t handle) NN_NOEXCEPT;
    nn::Result GetFileTimeStamp(uint64_t* pOutCreateTime, uint64_t* pOutAccessTime, uint64_t* pOutModifyTime, const char* pPath) NN_NOEXCEPT;
    nn::Result GetFileSize(int64_t* pOutSize, int32_t handle) NN_NOEXCEPT;
    nn::Result SetFileSize(int64_t size, int32_t handle) NN_NOEXCEPT;
    nn::Result OpenDirectory(int32_t* pOutHandle, const char* pPath, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT;
    nn::Result CloseDirectory(int32_t handle) NN_NOEXCEPT;
    nn::Result GetPriorityForDirectory(int32_t* pOutPriority, int32_t handle) NN_NOEXCEPT;
    nn::Result SetPriorityForDirectory(int32_t priority, int32_t handle) NN_NOEXCEPT;
    nn::Result CreateDirectory(const char* pPath) NN_NOEXCEPT;
    nn::Result DeleteDirectory(const char* pPath, bool recursively) NN_NOEXCEPT;
    nn::Result RenameDirectory(const char* pFromName, const char* pToName) NN_NOEXCEPT;
    nn::Result DirectoryExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT;
    nn::Result ReadDirectory(int64_t* pOutCount, nn::fs::DirectoryEntry* pOutBuffer, int64_t count, int32_t handle) NN_NOEXCEPT;
    nn::Result GetEntryCount(int64_t* pOutCount, int32_t handle) NN_NOEXCEPT;

private:
    static const nn::htclow::ChannelId HtclowChannelId = 0;

    HtcfsImpl() NN_NOEXCEPT
        : m_IsChannelInitialized(false)
    {
    }

    nn::Result InitializeChannel() NN_NOEXCEPT;
    void FinalizeChannel() NN_NOEXCEPT;

    nn::Result GetMaxProtocolVersion(int16_t* pOutVersion) NN_NOEXCEPT;
    nn::Result SetProtocolVersion(int16_t version) NN_NOEXCEPT;

    nn::Result SendToHtclow(const void* pBuffer, int64_t size) NN_NOEXCEPT;
    nn::Result ReceiveFromHtclow(void* pOutBuffer, int64_t size) NN_NOEXCEPT;

    nn::Result CheckResponseHeaderWithoutVersion(const Header& header, PacketType expectPacketType) NN_NOEXCEPT;
    nn::Result CheckResponseHeader(const Header& header, PacketType expectPacketType) NN_NOEXCEPT;
    nn::Result CheckResponseHeader(const Header& header, PacketType expectPacketType, int64_t expectBodySize) NN_NOEXCEPT;

    HeaderFactory m_Factory;
    nn::os::SdkMutex m_Mutex;
    nn::htclow::Channel m_Channel;
    bool m_IsChannelInitialized;
};

}}}
