﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/psc.h>

namespace nn { namespace gpio { namespace server {

// Gpio -> Pinmux -> GpioLow の順の依存でないと un-predictable glitches on output lines が発生する (NSBG-4121)
const nn::psc::PmModuleId GpioDependencies[]           = { nn::psc::PmModuleId_Pinmux };

enum StateEdge
{
    StateEdge_Unknown,
    StateEdge_FullAwakeToMinimumAwake,
    StateEdge_MinimumAwakeToSleepReady,
    StateEdge_SleepReadyToEssentialServicesSleepReady,
    StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake,
    StateEdge_EssentialServicesAwakeToMinimumAwake,
    StateEdge_MinimumAwakeToFullAwake,
};

inline StateEdge GetStateEdge(nn::psc::PmState previousState, nn::psc::PmState state)
{
    if (previousState == nn::psc::PmState_FullAwake &&
        state == nn::psc::PmState_MinimumAwake)
    {
        return StateEdge_FullAwakeToMinimumAwake;
    }
    else if (previousState == nn::psc::PmState_MinimumAwake &&
             state         == nn::psc::PmState_SleepReady)
    {
        return StateEdge_MinimumAwakeToSleepReady;
    }
    else if (previousState == nn::psc::PmState_SleepReady &&
             state         == nn::psc::PmState_EssentialServicesSleepReady)
    {
        return StateEdge_SleepReadyToEssentialServicesSleepReady;
    }
    else if (previousState == nn::psc::PmState_EssentialServicesSleepReady &&
             state         == nn::psc::PmState_EssentialServicesAwake)
    {
        return StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake;
    }
    else if (previousState == nn::psc::PmState_EssentialServicesAwake &&
             state         == nn::psc::PmState_MinimumAwake)
    {
        return StateEdge_EssentialServicesAwakeToMinimumAwake;
    }
    else if (previousState == nn::psc::PmState_MinimumAwake &&
             state         == nn::psc::PmState_FullAwake)
    {
        return StateEdge_MinimumAwakeToFullAwake;
    }
    else
    {
        return StateEdge_Unknown;
    }
}

// GPIO 用の 2つの PmModuleId を使用する IpcLoop 関数です。
template <typename ServerManagerT>
void IpcLoopForGpio(nn::psc::PmModule* pGpioPmModule,
                    nn::psc::PmState* pGpioPreviousState,
                    nn::psc::PmState* pGpioState,
                    nn::psc::PmFlagSet* pGpioFlags,
                    nn::psc::PmModule* pGpioLowPmModule,
                    nn::psc::PmState* pGpioLowPreviousState,
                    nn::psc::PmState* pGpioLowState,
                    nn::psc::PmFlagSet* pGpioLowFlags,
                    ServerManagerT* pServerManager) NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType gpioPmEventHolder;
    nn::os::InitializeMultiWaitHolder(
        &gpioPmEventHolder,
        pGpioPmModule->GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &gpioPmEventHolder,
        nn::psc::PmModuleId_Gpio);
    pServerManager->AddUserWaitHolder(&gpioPmEventHolder);

    nn::os::MultiWaitHolderType gpioLowPmEventHolder;
    nn::os::InitializeMultiWaitHolder(
        &gpioLowPmEventHolder,
        pGpioLowPmModule->GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &gpioLowPmEventHolder,
        nn::psc::PmModuleId_GpioLow);
    pServerManager->AddUserWaitHolder(&gpioLowPmEventHolder);

    while (NN_STATIC_CONDITION(true))
    {
        //NN_SDK_LOG("[gpio] Wait in IpcLoop\n");
        nn::os::MultiWaitHolderType* p = pServerManager->Wait();

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
            // SF processing
        case ServerManagerT::HipcSimpleAllInOneServerManager::AcceptTag:
        case ServerManagerT::HipcSimpleAllInOneServerManager::InvokeTag:
            //NN_SDK_LOG("[gpio] SF processing in IpcLoop\n");
            pServerManager->ProcessAuto(p);
            break;

        case nn::psc::PmModuleId_Gpio:
        {
            //NN_SDK_LOG("[gpio] Get PmGpio Event in IpcLoop\n");
            bool tryWaitResult = pGpioPmModule->GetEventPointer()->TryWait();
            NN_SDK_ASSERT(tryWaitResult);
            NN_UNUSED(tryWaitResult);
            pGpioPmModule->GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(pGpioPmModule->GetRequest(pGpioState, pGpioFlags));
            auto stateEdge = GetStateEdge(*pGpioPreviousState, *pGpioState);

            // IPC 不可状態に入る条件
            if (stateEdge == StateEdge_SleepReadyToEssentialServicesSleepReady)
            {
                //NN_SDK_LOG("[gpio] Exit IPC Loop\n");
                return;
            }
            else
            {
                pServerManager->AddUserWaitHolder(&gpioPmEventHolder);
                pGpioPmModule->Acknowledge(*pGpioState, nn::ResultSuccess());
                *pGpioPreviousState = *pGpioState;
                //NN_SDK_LOG("[gpio] Acknowledge by Gpio in IpcLoop\n");
                break;
            }
        }

        case nn::psc::PmModuleId_GpioLow:
        {
            //NN_SDK_LOG("[gpio] Get PmGpioLow Event in IpcLoop\n");
            bool tryWaitResult = pGpioLowPmModule->GetEventPointer()->TryWait();
            NN_SDK_ASSERT(tryWaitResult);
            NN_UNUSED(tryWaitResult);
            pGpioLowPmModule->GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(pGpioLowPmModule->GetRequest(pGpioLowState, pGpioLowFlags));
            auto stateEdge = GetStateEdge(*pGpioLowPreviousState, *pGpioLowState);

            // この Acknowledge の後、GPIO 側の EssentialServicesSleepReady でループを抜けるので、
            // ここで分岐が必要
            if (stateEdge == StateEdge_MinimumAwakeToSleepReady)
            {
                // Acknowledge はするものの、次で GPIO 側でループを抜けるため、Add はしない
                // ここで Add してしまうと、次の IPCLoop が正しく動かなくなる。
                pGpioLowPmModule->Acknowledge(*pGpioLowState, nn::ResultSuccess());
                *pGpioLowPreviousState = *pGpioLowState;
                //NN_SDK_LOG("[gpio] Acknowledge by GpioLow in IpcLoop\n");
                break;
            }
            else
            {
                // GPIO Low は IPC から抜ける条件を判定しない (Acknowledge するだけ)
                pServerManager->AddUserWaitHolder(&gpioLowPmEventHolder);
                pGpioLowPmModule->Acknowledge(*pGpioLowState, nn::ResultSuccess());
                *pGpioLowPreviousState = *pGpioLowState;
                //NN_SDK_LOG("[gpio] Acknowledge by GpioLow in IpcLoop\n");
                break;
            }
        }

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

}}} // nn::gpio::server
