﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>

#include <nn/gpio/gpio_IManager.sfdl.h> // for IGpioManager

#include "gpio_GetManagerByHipc.h"

namespace nn { namespace gpio {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::gpio::IManager> g_GpioManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

inline IPadSession* GetInterface(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession->_sessionHandle);
    return static_cast<nn::gpio::IPadSession*>(pSession->_sessionHandle);
}

}


void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_GpioManager);

        g_GpioManager = GetManagerByHipc();
    }

    g_InitializeCount++;
}

void InitializeWith(nn::sf::SharedPointer<nn::gpio::IManager> manager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_GpioManager = manager;

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_GpioManager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_GpioManager = nullptr;
    }
}

// [Gen2]
nn::Result OpenSession(GpioPadSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_ASSERT(g_GpioManager);
    nn::sf::SharedPointer<nn::gpio::IPadSession> session;
    NN_RESULT_DO(g_GpioManager->OpenSession2(&session, deviceCode.GetInternalValue()));

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
    pOutSession->_pEvent = nullptr;

    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSessionForDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_ASSERT(g_GpioManager);
    nn::sf::SharedPointer<nn::gpio::IPadSession> session;
    auto result = g_GpioManager->OpenSessionForDev(&session, static_cast<std::int32_t>(padNumber));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
    pOutSession->_pEvent = nullptr;
}

// [Gen1] TODO: Deprecate
// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_ASSERT(g_GpioManager);
    nn::sf::SharedPointer<nn::gpio::IPadSession> session;
    auto result = g_GpioManager->OpenSession(&session, static_cast<std::int32_t>(pad));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
    pOutSession->_pEvent = nullptr;
}

// [Gen1] TODO: Deprecate
void OpenSessionForTest(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_ASSERT(g_GpioManager);
    nn::sf::SharedPointer<nn::gpio::IPadSession> session;
    auto result = g_GpioManager->OpenSessionForTest(&session, static_cast<std::int32_t>(padNumber));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
    pOutSession->_pEvent = nullptr;
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);

    // Unbind されていないなら Unbind する
    if(pSession->_pEvent != nullptr)
    {
        nn::gpio::UnbindInterrupt(pSession);
    }

    nn::sf::ReleaseSharedObject(GetInterface(pSession));
}


// Session がもつ各機能のラップ版
void SetDirection(GpioPadSession* pSession, Direction direction) NN_NOEXCEPT
{

    auto result = GetInterface(pSession)->SetDirection(static_cast<int32_t>(direction));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

GpioValue GetValue(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int32_t outValue;

    auto result = GetInterface(pSession)->GetValue(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<nn::gpio::GpioValue>(outValue);
}

void SetValue(GpioPadSession* pSession, GpioValue value) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetValue(static_cast<int32_t>(value));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void SetValueForSleepState(GpioPadSession* pSession, GpioValue value) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetValueForSleepState(static_cast<int32_t>(value));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

Direction GetDirection(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int32_t outDirection;

    auto result = GetInterface(pSession)->GetDirection(&outDirection);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<nn::gpio::Direction>(outDirection);
}

void SetInterruptMode(GpioPadSession* pSession, InterruptMode mode) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetInterruptMode(static_cast<int32_t>(mode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

InterruptMode GetInterruptMode(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int32_t outMode;

    auto result = GetInterface(pSession)->GetInterruptMode(&outMode);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<nn::gpio::InterruptMode>(outMode);
}

void SetInterruptEnable(GpioPadSession* pSession, bool enable) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetInterruptEnable(static_cast<int32_t>(enable));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

bool GetInterruptEnable(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    bool isEnable;

    auto result = GetInterface(pSession)->GetInterruptEnable(&isEnable);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isEnable;
}

InterruptStatus GetInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int32_t outStatus;

    auto result = GetInterface(pSession)->GetInterruptStatus(&outStatus);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<nn::gpio::InterruptStatus>(outStatus);
}

void ClearInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->ClearInterruptStatus();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

// 内部で用意されている SystemEvent を Bind してそれを渡す
Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;

    NN_RESULT_DO(GetInterface(pSession)->BindInterrupt(&sfHandle));

    nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                              sfHandle.GetOsHandle(),
                                              sfHandle.IsManaged(),
                                              nn::os::EventClearMode_ManualClear);
    sfHandle.Detach();

    // UnbindInterrupt で SystemEvent を破棄できるように SessionHandle のメンバ変数にポインタを保存
    pSession->_pEvent = pEvent;

    return nn::ResultSuccess();
}

void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT
{
    // SessionHandle が持つpEvent が null の場合は Bind されてないのに呼び出されているので ASSERT
    NN_SDK_ASSERT(pSession->_pEvent != nullptr);

    auto result = GetInterface(pSession)->UnbindInterrupt();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Bind されていたイベントを破棄する。
    nn::os::DestroySystemEvent(pSession->_pEvent);

    pSession->_pEvent = nullptr;
}

void SetDebounceEnabled(GpioPadSession* pSession, bool isEnable) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetDebounceEnabled(isEnable);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

bool GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    bool isEnable;

    auto result = GetInterface(pSession)->GetDebounceEnabled(&isEnable);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return isEnable;
}

void SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetDebounceTime(msecTime);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int time;

    auto result = GetInterface(pSession)->GetDebounceTime(&time);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return time;
}

// [Gen2]
nn::Result IsWakeEventActive(bool* pOutIsActive, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsActive);

    // 出力用の変数
    bool isActive;

    NN_RESULT_DO(g_GpioManager->IsWakeEventActive2(&isActive, deviceCode.GetInternalValue()));

    *pOutIsActive = isActive;
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
bool IsWakeEventActive(GpioPadName name) NN_NOEXCEPT
{
    // 出力用の変数
    bool isActive;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_GpioManager->IsWakeEventActive(&isActive, static_cast<std::int32_t>(name)));

    return isActive;
}

// [Gen2] NOT SUPPORTED
// [Gen1] TODO: Deprecate
WakeBitFlag GetWakeEventActiveFlagSet() NN_NOEXCEPT
{
    WakeBitFlag wakeBitFlag;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_GpioManager->GetWakeEventActiveFlagSet(&wakeBitFlag));

    return wakeBitFlag;
}

// [Gen2]
nn::Result SetWakeEventActiveFlagSetForDebug(nn::DeviceCode deviceCode, bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(g_GpioManager->SetWakeEventActiveFlagSetForDebug2(deviceCode.GetInternalValue(), isEnabled));
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
void SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool isEnabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_GpioManager->SetWakeEventActiveFlagSetForDebug(static_cast<std::int32_t>(name), isEnabled));
}

void SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_GpioManager->SetWakePinDebugMode(static_cast<std::int32_t>(mode)));
}

}}
