﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/gpio/driver/gpio_IGpioDriver.h>
#include <nn/gpio/driver/gpio_Pad.h>
#include <nn/gpio/driver/detail/gpio_PadSessionImpl.h>

namespace nn {
namespace gpio {
namespace driver {

bool Pad::IsAnySessionBoundToInterrupt() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(GetDriver().SafeCastTo<IGpioDriver>().GetInterruptControlMutex(*this).IsLockedByCurrentThread(),
        "Pad::IsAnySessionBoundToInterrupt() must be called after obtaining lock for interrupt enable mutex of the pad.\n");

    bool isBound = false;
    ForEachSession(
        [&isBound](const nn::ddsf::ISession* pSession) NN_NOEXCEPT -> bool
        {
            auto& sessionImpl = pSession->SafeCastTo<detail::PadSessionImpl>();
            if ( sessionImpl.IsInterruptBound() )
            {
                isBound = true;
                return false; // Break ForEach loop
            }
            return true;
        }
    );
    return isBound;
}

void Pad::SignalInterruptBoundEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(GetDriver().SafeCastTo<IGpioDriver>().GetInterruptControlMutex(*this).IsLockedByCurrentThread(),
        "Pad::SignalInterruptBoundEvent() must be called after obtaining lock for interrupt enable mutex of the pad.\n");

    ForEachSession(
        [](nn::ddsf::ISession* pSession) NN_NOEXCEPT -> bool
        {
            auto& sessionImpl = pSession->SafeCastTo<detail::PadSessionImpl>();
            if ( sessionImpl.IsInterruptBound() )
            {
                sessionImpl.SignalInterruptBoundEvent();
            }
            return true;
        }
    );
}

} // driver
} // gpio
} // nn
