﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gpio/detail/gpio_Log.h>

#include "gpio_RegAccessor-soc.tegra.h"
#include "gpio_RegAccessorPinmux-soc.tegra.h"
#include "gpio_DdUtil.h"

namespace nn{
namespace gpio{
namespace driver{
namespace detail{

namespace
{
// PINMUX レジスタの Field Name の列挙体
enum PinmuxFieldName
{
    PinmuxFieldName_PINMUX0  = 0,
    PinmuxFieldName_PINMUX1,
    PinmuxFieldName_PUPD2,
    PinmuxFieldName_PUPD3,
    PinmuxFieldName_TRISTATE,
    PinmuxFieldName_PARK,
    PinmuxFieldName_E_INPUT,
    PinmuxFieldName_LOCK,
    PinmuxFieldName_E_LPDR,
    PinmuxFieldName_E_OD,
    PinmuxFieldName_E_SCHMT,
};

enum PinmuxFieldBitFlag
{
    PinmuxFieldBitFlag_PINMUX   = (1 << PinmuxFieldName_PINMUX1) | (1 << PinmuxFieldName_PINMUX0),
    PinmuxFieldBitFlag_PUPD     = (1 << PinmuxFieldName_PUPD3)   | (1 << PinmuxFieldName_PUPD2),
    PinmuxFieldBitFlag_TRISTATE = (1 << PinmuxFieldName_TRISTATE),
    PinmuxFieldBitFlag_PARK     = (1 << PinmuxFieldName_PARK),
    PinmuxFieldBitFlag_E_INPUT  = (1 << PinmuxFieldName_E_INPUT),
    PinmuxFieldBitFlag_LOCK     = (1 << PinmuxFieldName_LOCK),
    PinmuxFieldBitFlag_E_LPDR   = (1 << PinmuxFieldName_E_LPDR),
    PinmuxFieldBitFlag_E_OD     = (1 << PinmuxFieldName_E_OD),
    PinmuxFieldBitFlag_E_SCHMT  = (1 << PinmuxFieldName_E_SCHMT),
};

}

// GPIO ライブラリで使用するために Pinmux を設定する関数
void PinmuxRegAccessor::SetGpioPinmux(InternalGpioPadNumber pad, Direction direction)
{
    nn::Bit32* address = nullptr;;

    // TORIAEZU: Pinmux 関連はどこかで一括設定をするようになったら削除する。
    // それまでは内部でサポートしているパッドの分だけを列挙しておく。
    switch(pad)
    {
    case InternalGpioPadNumber_GPIO_PE1:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x30a8);
        break;

    case InternalGpioPadNumber_GPIO_PE4:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x30b4);
        break;

    case InternalGpioPadNumber_GPIO_PE7:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x324c);
        break;

    case InternalGpioPadNumber_GPIO_PH4:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x31c4);
        break;

    case InternalGpioPadNumber_GPIO_PK0:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3254);
        break;

    case InternalGpioPadNumber_GPIO_PK1:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3258);
        break;

    case InternalGpioPadNumber_GPIO_PK2:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x325c);
        break;

    case InternalGpioPadNumber_GPIO_PK3:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3260);
        break;

    case InternalGpioPadNumber_GPIO_PK7:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3270);
        break;

    case InternalGpioPadNumber_GPIO_PZ4:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x328c);
        break;

    case InternalGpioPadNumber_GPIO_PBB3:
        address = reinterpret_cast<nn::Bit32*>(m_ApbMiscVirtualAddress + 0x318c);
        break;

    default :
        // TORIAEZU: システム全体で Pinmux がセットされるまではこのログを出しておく。
        NN_DETAIL_GPIO_INFO("Pinmux for Pad(%d) is not set by GPIO Driver\n", pad);
        address = nullptr;
    }

    // Pull-up / Pull-down を OFF にして、TRISTATE も OFF に。(GPIO_OE でコントロールするため)
    //(Tegra_K1_TRM_DP06905001v02p.pdf p.268 Table 28 参照)
    if(address != nullptr)
    {
        switch (pad)
        {
        case InternalGpioPadNumber_GPIO_PE1:
            // PE1 の時は Pullup に設定する
            WriteMasked32(address, 0x02 << PinmuxFieldName_PUPD2, PinmuxFieldBitFlag_TRISTATE | PinmuxFieldBitFlag_PUPD);
            break;

        case InternalGpioPadNumber_GPIO_PK7:
            // PK7 の時は Pullup に設定する
            WriteMasked32(address, 0x02 << PinmuxFieldName_PUPD2, PinmuxFieldBitFlag_TRISTATE | PinmuxFieldBitFlag_PUPD);
            break;

        default:
            WriteMasked32(address, 0x00, PinmuxFieldBitFlag_TRISTATE | PinmuxFieldBitFlag_PUPD);
        }

        if (direction == Direction_Output)
        {
            // E_INPUT = 0 (input receiver disabled)
            SetBit<Bit32>(address, 0, PinmuxFieldName_E_INPUT);
        }
        else
        {
            // E_INPUT = 1 (input receiver enabled)
            SetBit<Bit32>(address, 1, PinmuxFieldName_E_INPUT);
        }
        DummyRead(address);
    }
}

} // detail
} // driver
} // gpio
} // nn
