﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/account/account_Api.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_TypesForSaveDataManagement.h>
#include <nn/fs/fs_Utility.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_Result.h>
#include <nn/am/am_Shim.h>
#include <nn/arp/arp_Result.h>
#include <nn/err/err_Api.h>

#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

namespace {
    Result EnsureSaveDataImpl(int64_t* pOutRequiredSize, const nn::account::Uid& user) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(user);
        *pOutRequiredSize = 0;
#else
        auto result = nn::am::GetApplicationFunctions()->EnsureSaveData(pOutRequiredSize, user);
        if( ns::ResultProcessNotFound::Includes(result) || arp::ResultNotRegistered::Includes(result) )
        {
            // TORIAEZU: アプリケーション管理データが取得できない場合は何もせず、MountSaveData() での暗黙作成に任せる
            *pOutRequiredSize = 0;
        }
        else
        {
            NN_RESULT_DO(result);
        }
#endif
        NN_RESULT_SUCCESS;
    }

    Result CreateCacheStorageImpl(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(index);
        NN_UNUSED(cacheStorageSize);
        NN_UNUSED(cacheStorageJournalSize);
        *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Nand;
        *pOutRequiredSize = 0;
        return nn::fs::ResultUnsupportedOperation();
#else
        auto result = nn::am::GetApplicationFunctions()->CreateCacheStorage(pOutRequiredSize, reinterpret_cast<int32_t*>(pOutCacheStorageTargetMedia), index, cacheStorageSize, cacheStorageJournalSize);
        if( ns::ResultProcessNotFound::Includes(result) || arp::ResultNotRegistered::Includes(result) )
        {
            // TORIAEZU: アプリケーション管理データが取得できない場合は何もせず、MountSaveData() での暗黙作成に任せる
            *pOutRequiredSize = 0;
        }
        else
        {
            NN_RESULT_DO(result);
        }
        NN_RESULT_SUCCESS;
#endif
    }

    Result ExtendSaveDataImpl(int64_t* pOutRequiredSize, nn::fs::SaveDataType saveDataType, const nn::account::Uid& user, int64_t saveDataSize, int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(saveDataType);
        NN_UNUSED(user);
        NN_UNUSED(saveDataSize);
        NN_UNUSED(saveDataJournalSize);
        *pOutRequiredSize = 0;
#else
        auto result = nn::am::GetApplicationFunctions()->ExtendSaveData(pOutRequiredSize, static_cast<uint8_t>(saveDataType), user, saveDataSize, saveDataJournalSize);
        if( ns::ResultProcessNotFound::Includes(result) || arp::ResultNotRegistered::Includes(result) )
        {
            // TORIAEZU: アプリケーション管理データが取得できない場合は何もしない
            *pOutRequiredSize = 0;
        }
        else if( fs::ResultTargetLocked::Includes(result) )
        {
            // マウント中の拡張はデフォルトで Abort にする
            auto abortSpecifier = GetCurrentThreadFsContext()->HandleResult(result);
            switch (abortSpecifier)
            {
            case AbortSpecifier::ReturnResult:
                return result;
            default:
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        else
        {
            NN_RESULT_DO(result);
        }
#endif
        NN_RESULT_SUCCESS;
    }

    Result GetSaveDataSizeImpl(int64_t* outSaveDataSize, int64_t* outSaveDataJournalSize, nn::fs::SaveDataType saveDataType, const nn::account::Uid& user) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(outSaveDataSize != nullptr, fs::ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(outSaveDataJournalSize != nullptr, fs::ResultInvalidArgument());

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(saveDataType);
        NN_UNUSED(user);
        *outSaveDataSize = 0;
        *outSaveDataJournalSize = 0;
#else
        NN_RESULT_DO(nn::am::GetApplicationFunctions()->GetSaveDataSize(outSaveDataSize, outSaveDataJournalSize, static_cast<uint8_t>(saveDataType), user));
#endif
        NN_RESULT_SUCCESS;
    }

    Result ShowLibraryAppletDataErase(const account::Uid& uid, CacheStorageTargetMedia media, int64_t size, nn::Result throwResult, SaveDataType type) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(uid);
        NN_UNUSED(media);
        NN_UNUSED(size);
        NN_UNUSED(throwResult);
        NN_UNUSED(type);
#else
        static const uint16_t VersionMajor = 0;
        static const uint16_t VersionMinor = 2;// type を追加したので 1 -> 2 に更新

        // 呼び出しパラメータ
        UiParameterForStorageManagement param;
        param.needSize = size;
        param.media = static_cast<UiParameterForStorageManagement::Media>(media);
        param.uid = uid;
        if(type == SaveDataType::Account)
        {
            param.type = UiParameterForStorageManagement::DataType_AccountSave;
        }
        else if(type == SaveDataType::Cache)
        {
            param.type = UiParameterForStorageManagement::DataType_CacheStorage;
        }
        else
        {
            // 来ないはず
            return nn::fs::ResultInvalidArgument();
        }
        // LA 共通パラメータ
        nn::la::CommonArgumentsWriter commonArg(VersionMajor, VersionMinor);

        // LA 呼び出し
        int rval;
        size_t outSize;
        auto result = nn::la::StartLibraryAppletEasy(
            nn::applet::AppletId_LibraryAppletDataErase, commonArg,
            &param, sizeof(param),
            &rval, sizeof(rval), &outSize);
        NN_RESULT_THROW_UNLESS(
            result.IsSuccess() && outSize >= sizeof(rval) && rval == 0,
            throwResult);
#endif
        NN_RESULT_SUCCESS;
    }
}

    Result MountSaveData(const char* name, const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        return MountSaveData(name, ConvertAccountUidToFsUserId(user));
    }

    Result MountSaveData(const char* name, nn::ApplicationId applicationId, const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        nn::ncm::ApplicationId appId = { applicationId.value };
        return MountSaveData(name, appId, ConvertAccountUidToFsUserId(user));
    }

    Result MountSaveDataReadOnly(const char* name, const nn::ApplicationId applicationId, const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        nn::ncm::ApplicationId appId = { applicationId.value };
        return MountSaveDataReadOnly(name, appId, ConvertAccountUidToFsUserId(user));
    }

    bool IsSaveDataExisting(const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        return IsSaveDataExisting(ConvertAccountUidToFsUserId(user));
    }

    bool IsSaveDataExisting(const nn::ApplicationId applicationId, const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        nn::ncm::ApplicationId appId = { applicationId.value };
        return IsSaveDataExisting(appId, ConvertAccountUidToFsUserId(user));
    }

    Result EnsureSaveData(const nn::account::Uid& user) NN_NOEXCEPT
    {
        auto ensure = [=]() NN_NOEXCEPT -> Result
        {
            if (user != nn::account::InvalidUid)
            {
                CheckUid(user);
            }

            int64_t requiredSize;
            NN_RESULT_DO(EnsureSaveDataImpl(&requiredSize, user));
            if (requiredSize != 0)
            {
                // 空き容量不足のハンドリングとして削除アプレット呼び出し
                NN_RESULT_DO(ShowLibraryAppletDataErase(user, CacheStorageTargetMedia_Nand, requiredSize, ResultUsableSpaceNotEnoughForSaveData(), SaveDataType::Account));

                // 容量不足解消後に再トライ
                NN_RESULT_DO(EnsureSaveDataImpl(&requiredSize, user));
                if (requiredSize != 0)
                {
                    // コーナーケースで容量不足になる場合があるため、特殊なエラーを表示
                    err::ShowError(ResultUsableSpaceNotEnoughForSaveDataEvenAssistanceSuccess());

                    // アプリはセーブデータ作成失敗をハンドリングする必要がある
                    NN_RESULT_THROW(ResultUsableSpaceNotEnoughForSaveData());
                }

                // 容量不足が解消された
            }
            NN_RESULT_SUCCESS;
        };

        NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(ensure(),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID, user._data[0], user._data[1]));
        NN_RESULT_SUCCESS;
    }

    Result ExtendSaveData(const nn::account::Uid& user, int64_t saveDataSize, int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        auto extend = [=]() NN_NOEXCEPT -> Result
        {
            AssertCheckUid(user);

            int64_t requiredSize;
            NN_RESULT_DO(ExtendSaveDataImpl(&requiredSize, nn::fs::SaveDataType::Account, user, saveDataSize, saveDataJournalSize));
            if (requiredSize != 0)
            {
                // 空き容量不足のハンドリングとして削除アプレット呼び出し
                NN_RESULT_DO(ShowLibraryAppletDataErase(user, CacheStorageTargetMedia_Nand, requiredSize, ResultUsableSpaceNotEnoughForSaveData(), SaveDataType::Account));

                // 容量不足解消後に再トライ
                NN_RESULT_DO(ExtendSaveDataImpl(&requiredSize, nn::fs::SaveDataType::Account, user, saveDataSize, saveDataJournalSize));
                if (requiredSize != 0)
                {
                    // コーナーケースで容量不足になる場合があるため、特殊なエラーを表示
                    err::ShowError(ResultUsableSpaceNotEnoughForSaveDataEvenAssistanceSuccess());

                    // アプリはセーブデータ作成失敗をハンドリングする必要がある
                    NN_RESULT_THROW(ResultUsableSpaceNotEnoughForSaveData());
                }

                // 容量不足が解消された
            }
            NN_RESULT_SUCCESS;
        };

        NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(extend(),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_EXTENDSAVEDATA(user, saveDataSize, saveDataJournalSize)));
        NN_RESULT_SUCCESS;
    }

#if 0
    Result ExtendDeviceSaveData(int64_t saveDataSize, int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        auto invalidUser = nn::account::InvalidUid;
        int64_t requiredSize;
        NN_FS_RESULT_DO(ExtendSaveDataImpl(&requiredSize, nn::fs::SaveDataType::Device, invalidUser, saveDataSize, saveDataJournalSize));
        if (requiredSize != 0)
        {
            // 空き容量不足のハンドリングとして削除アプレット呼び出し
            NN_RESULT_DO(ShowLibraryAppletDataErase(invalidUser, requiredSize));

            // 容量不足解消後に再トライ
            NN_FS_RESULT_DO(ExtendSaveDataImpl(&requiredSize, nn::fs::SaveDataType::Device, invalidUser, saveDataSize, saveDataJournalSize));
            if (requiredSize != 0)
            {
                // コーナーケースで容量不足になる場合があるため、特殊なエラーを表示
                err::ShowError(ResultUsableSpaceNotEnoughForSaveDataEvenAssistanceSuccess());

                // アプリはセーブデータ作成失敗をハンドリングする必要がある
                NN_RESULT_THROW(ResultUsableSpaceNotEnoughForSaveData());
            }

            // 容量不足が解消された
        }
        NN_RESULT_SUCCESS;
    }
#endif

    Result GetSaveDataSize(int64_t* outSaveDataSize, int64_t* outSaveDataJournalSize, const nn::account::Uid& user) NN_NOEXCEPT
    {
        AssertCheckUid(user);
        NN_FS_RESULT_DO(
            NN_DETAIL_FS_ACCESS_LOG(
                GetSaveDataSizeImpl(outSaveDataSize, outSaveDataJournalSize, nn::fs::SaveDataType::Account, user)
                , nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_GETSAVEDATASIZE(user)
            )
        );
        NN_RESULT_SUCCESS;
    }

#if 0
    Result GetDeviceSaveDataSize(int64_t* outSaveDataSize, int64_t* outSaveDataJournalSize) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(GetSaveDataSizeImpl(outSaveDataSize, outSaveDataJournalSize, nn::fs::SaveDataType::Device, nn::account::InvalidUid));
        NN_RESULT_SUCCESS;
    }
#endif

Result CreateCacheStorage(int index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize) NN_NOEXCEPT
{
    uint16_t internalIndex = static_cast<uint16_t>(index);

    auto create = [=]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultInvalidArgument());
        int64_t requiredSize;
        CacheStorageTargetMedia cacheStorageTargetMedia;
        NN_RESULT_DO(CreateCacheStorageImpl(&requiredSize, &cacheStorageTargetMedia, internalIndex, cacheStorageSize, cacheStorageJournalSize));

        if (requiredSize != 0)
        {
            // 空き容量不足のハンドリングとして削除アプレット呼び出し
            NN_RESULT_DO(ShowLibraryAppletDataErase(nn::account::InvalidUid, cacheStorageTargetMedia, requiredSize, ResultUsableSpaceNotEnoughForCacheStorage(), SaveDataType::Cache));

            // 容量不足解消後に再トライ
            NN_RESULT_DO(CreateCacheStorageImpl(&requiredSize, &cacheStorageTargetMedia, internalIndex, cacheStorageSize, cacheStorageJournalSize));
            if (requiredSize != 0)
            {
                // コーナーケースで容量不足になる場合があるため、特殊なエラーを表示
                err::ShowError(ResultUsableSpaceNotEnoughForSaveDataEvenAssistanceSuccess());

                // アプリはセーブデータ作成失敗をハンドリングする必要がある
                NN_RESULT_THROW(ResultUsableSpaceNotEnoughForCacheStorage());
            }

            // 容量不足が解消された
        }
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(create(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATECACHESTORAGE(internalIndex, cacheStorageSize, cacheStorageJournalSize)));
    NN_RESULT_SUCCESS;
}

}}
