﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/service/friends_Server.h>
#include <nn/friends/detail/ipc/friends_ServiceConfig.h>
#include <nn/friends/detail/ipc/friends_ServiceName.h>
#include <nn/friends/detail/service/friends_Capability.h>
#include <nn/friends/detail/service/friends_Service.h>
#include <nn/friends/detail/service/core/friends_FileSystem.h>
#include <nn/friends/detail/service/core/friends_EventWatcher.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskThread.h>
#include <nn/friends/detail/service/core/friends_ForegroundTaskThread.h>
#include <nn/friends/detail/service/core/friends_NotificationReceiver.h>
#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/service/core/friends_PlayLogManager.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace friends { namespace service {

namespace
{
    const int ParallelExecutionCountMax = 4;

    nn::os::ThreadType g_Threads[ParallelExecutionCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStacks[ParallelExecutionCountMax][16 * 1024];

    struct FriendServiceHipcSimpleAllInOneServerManagerOption
    {
        // これ以上のサイズのデータは、[BufferTransferMode(BufferTransferMode.MapAlias)] でマップ転送に切り替えること。
        static const size_t PointerTransferBufferSize = 2560;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = 64;
        static const int ObjectInSubDomainCountMax = 16;
    };

    class FriendServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<64, detail::ipc::ServiceCount,
            FriendServiceHipcSimpleAllInOneServerManagerOption>
    {
    };

    FriendServiceServerManager* g_ServerManager = nullptr;
    std::aligned_storage<sizeof (FriendServiceServerManager)>::type g_ServerManagerStorage;

    bool g_IsBackgroundProcessingEnabled = true;
}

namespace
{
    void RegisterService(int index, const char* serviceName, Bit32 capability) NN_NOEXCEPT
    {
        detail::service::InitializeServiceCreator(index, serviceName, capability);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager->RegisterObjectForPort(detail::service::GetServiceCreator(index),
            detail::ipc::SessionCountMax, serviceName));
    }

    void IpcDispatcher(void*) NN_NOEXCEPT
    {
        g_ServerManager->LoopAuto();
    }

    void CheckBackgroundProcessingEnabled() NN_NOEXCEPT
    {
        bool isEnabled;

        if (nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, 1, "friends", "background_processing") == 1)
        {
            g_IsBackgroundProcessingEnabled = isEnabled;
        }
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] Start service...\n");

    g_ServerManager = new (&g_ServerManagerStorage) FriendServiceServerManager;

    // friend:a
    RegisterService(0, detail::ipc::ServiceNameForAdministrator,
        detail::service::Capability::Flag_All);

    // friend:u
    RegisterService(1, detail::ipc::ServiceNameForGeneric,
        detail::service::Capability::Flag_Generic);

    // friend:v
    RegisterService(2, detail::ipc::ServiceNameForViewer,
        detail::service::Capability::Flag_Generic | detail::service::Capability::Flag_View);

    // friend:m
    RegisterService(3, detail::ipc::ServiceNameForManager,
        detail::service::Capability::Flag_Generic | detail::service::Capability::Flag_View | detail::service::Capability::Flag_Management);

    // friend:s
    RegisterService(4, detail::ipc::ServiceNameForSystem,
        detail::service::Capability::Flag_Generic | detail::service::Capability::Flag_System);

    g_ServerManager->Start();

    for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i],
            IpcDispatcher, nullptr, g_ThreadStacks[i], sizeof (g_ThreadStacks[i]), NN_SYSTEM_THREAD_PRIORITY(friends, IpcServer)));

        nn::os::SetThreadNamePointer(&g_Threads[i], NN_SYSTEM_THREAD_NAME(friends, IpcServer));
        nn::os::StartThread(&g_Threads[i]);
    }

    detail::service::core::EventWatcher::Iniaialize();

    CheckBackgroundProcessingEnabled();

    if (g_IsBackgroundProcessingEnabled)
    {
        detail::service::core::EventWatcher::Start();
        detail::service::core::BackgroundTaskThread::Start();
        detail::service::core::NotificationReceiver::Start();
    }
    else
    {
        NN_DETAIL_FRIENDS_WARN("\033[31m[friends] Background processing is disabled.\033[m\n");
    }

    detail::service::core::ForegroundTaskThread::Start();

    NN_DETAIL_FRIENDS_INFO("[friends] Start service... done!\n");
}

void StopServer() NN_NOEXCEPT
{
    g_ServerManager->RequestStop();

    for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
    {
        nn::os::DestroyThread(&g_Threads[i]);
    }

    g_ServerManager->~FriendServiceServerManager();
    g_ServerManager = nullptr;

    detail::service::FinalizeServiceCreators();

    detail::service::core::ForegroundTaskThread::Stop();

    if (g_IsBackgroundProcessingEnabled)
    {
        detail::service::core::NotificationReceiver::Stop();
        detail::service::core::BackgroundTaskThread::Stop();
        detail::service::core::EventWatcher::Stop();
    }

    NN_DETAIL_FRIENDS_INFO("[friends] End service.\n");
}

void NotifyBgtcEventTriggered(bool isInHalfAwake) NN_NOEXCEPT
{
    NN_UNUSED(isInHalfAwake);

    detail::service::core::BackgroundTaskThread::NotifyMinimumAwaked();
}

void NotifySystemFullAwaked() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] Resume.\n");

    detail::service::core::BackgroundTaskThread::NotifySleepAwaked();

    detail::service::core::PlayLogManager::GetInstance().NotifySleepAwaked();
    detail::service::core::BackgroundTaskManager::GetInstance().NotifySleepAwaked();
    detail::service::core::UserPresenceManager::GetInstance().NotifySleepAwaked();
}

void NotifySystemMinimumAwaked(bool isWakeSequence) NN_NOEXCEPT
{
    // friends は半起床中にサービスを実行しないため、 MinimumAwake 遷移でスリープ状態扱いとする。
    if (!isWakeSequence)
    {
        NN_DETAIL_FRIENDS_INFO("[friends] Suspend.\n");

        detail::service::core::BackgroundTaskThread::NotifySleep();

        detail::service::core::PlayLogManager::GetInstance().NotifySleep();
        detail::service::core::BackgroundTaskManager::GetInstance().NotifySleep();
        detail::service::core::UserPresenceManager::GetInstance().NotifySleep();
    }
}

void NotifySystemSleepReady() NN_NOEXCEPT
{
}

void NotifySystemShutdownReady() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] Shutdown.\n");
}

void WaitForSleepReady() NN_NOEXCEPT
{
}

void WaitForShutdownReady() NN_NOEXCEPT
{
}

}}}
