﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/json/friends_JsonPath.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

JsonPath::JsonPath() NN_NOEXCEPT :
    m_Depth(0),
    m_PathLength(1)
{
    m_Nodes[0].type = NodeType_Root;
    m_Nodes[0].validJsonPath = true;
    m_Nodes[0].head = 0;
    m_Nodes[0].arrayCount = 0;

    m_Path[0] = '$';
    m_Path[1] = '\0';
}

bool JsonPath::Compare(const char* path) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return (std::strncmp(m_Path, path, sizeof (m_Path)) == 0);
}

bool JsonPath::Match(const char* path) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    const char* p1 = path;
    const char* p2 = m_Path;

    int nodeCount = 0;

    while (*p1)
    {
        const char* s1 = p1;
        const char* s2 = p2;
        const char* e1 = p1;
        const char* e2 = p2;
        const char* w = nullptr;

        // ノード名の末尾まで移動する。(e1, e2)
        while (*e1 && *e1 != '.')
        {
            if (*e1 == '*')
            {
                // 1 ノードにワイルドカード文字が複数回出てはいけない。
                NN_SDK_REQUIRES(!w);
                w = e1;
            }
            e1++;
        }
        while (*e2 && *e2 != '.')
        {
            e2++;
        }

        // 次のノードの先頭位置を取得する。
        p1 = e1 + ((*e1 == '.') ? 1 : 0);
        p2 = e2 + ((*e2 == '.') ? 1 : 0);

        if (w)
        {
            // ワイルドカード文字出現まで文字列比較を行う。
            if (s1 < w)
            {
                while (s1 != w && s2 != e2 && *s1 == *s2)
                {
                    s1++;
                    s2++;
                }
                if (s1 != w)
                {
                    return false;
                }
            }
            if (e1 > w)
            {
                while (e1 != w && s2 != e2 && *e1 == *e2)
                {
                    e1--;
                    e2--;
                }
                if (e1 != w)
                {
                    return false;
                }
            }
        }
        else
        {
            while (s1 != e1 && s2 != e2 && *s1 == *s2)
            {
                s1++;
                s2++;
            }
            if (s1 != e1 || s2 != e2)
            {
                return false;
            }
        }

        nodeCount++;
    }

    return (*p2 == '\0');
}

const char* JsonPath::ToString() const NN_NOEXCEPT
{
    return m_Path;
}

bool JsonPath::IsValid() const NN_NOEXCEPT
{
    return m_Nodes[m_Depth].validJsonPath;
}

bool JsonPath::TryPushObject() NN_NOEXCEPT
{
    if (m_Depth == NodeDepthMax)
    {
        return false;
    }

    UpdateCurrent();

    m_Depth++;
    m_Nodes[m_Depth].type = NodeType_Object;
    m_Nodes[m_Depth].validJsonPath = m_Nodes[m_Depth - 1].validJsonPath;
    m_Nodes[m_Depth].head = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;

    return true;
}

bool JsonPath::TryPushArray() NN_NOEXCEPT
{
    if (m_Depth == NodeDepthMax)
    {
        return false;
    }

    UpdateCurrent();

    m_Depth++;
    m_Nodes[m_Depth].type = NodeType_Array;
    m_Nodes[m_Depth].validJsonPath = m_Nodes[m_Depth - 1].validJsonPath;
    m_Nodes[m_Depth].head = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;

    uint16_t remainLength = PathLengthMax - m_PathLength;

    if (remainLength < 2)
    {
        InvalidateJsonPath();
    }
    else
    {
        m_Path[m_PathLength++] = '[';
        m_Path[m_PathLength++] = ']';
        m_Path[m_PathLength] = '\0';
    }

    return true;
}

bool JsonPath::TryPushKey(const char* key, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);

    if (m_Depth == NodeDepthMax)
    {
        return false;
    }

    UpdateCurrent();

    m_Depth++;
    m_Nodes[m_Depth].type = NodeType_Key;
    m_Nodes[m_Depth].validJsonPath = m_Nodes[m_Depth - 1].validJsonPath;
    m_Nodes[m_Depth].head = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;

    uint16_t remainLength = PathLengthMax - m_PathLength;

    if (length + 1 > remainLength || isOverflowed)
    {
        InvalidateJsonPath();
    }
    else
    {
        m_Path[m_PathLength++] = '.';

        for (size_t i = 0; i < length; i++)
        {
            char c = key[i];

            if (c == '\0')
            {
                break;
            }

            m_Path[m_PathLength++] = c;
        }

        m_Path[m_PathLength] = '\0';
    }

    return true;
}

void JsonPath::UpdateCurrent() NN_NOEXCEPT
{
    if (m_Nodes[m_Depth].type != NodeType_Array)
    {
        return;
    }

    uint16_t remainLength = PathLengthMax - m_Nodes[m_Depth].head;

    uint16_t length = static_cast<uint16_t>
        (nn::util::SNPrintf(m_Path + m_Nodes[m_Depth].head, remainLength + 1, "[%u]", m_Nodes[m_Depth].arrayCount));

    if (length > remainLength)
    {
        InvalidateJsonPath();
    }
    else
    {
        m_PathLength = m_Nodes[m_Depth].head + length;
    }

    m_Nodes[m_Depth].arrayCount++;
}

void JsonPath::Pop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Depth > 0);

    if (m_Nodes[m_Depth].type == NodeType_Array)
    {
        uint16_t remainLength = PathLengthMax - m_Nodes[m_Depth].head;

        if (remainLength < 2)
        {
            InvalidateJsonPath();
        }
        else
        {
            m_PathLength = m_Nodes[m_Depth].head;
            m_Path[m_PathLength++] = '[';
            m_Path[m_PathLength++] = ']';
            m_Path[m_PathLength] = '\0';

            m_Nodes[m_Depth].validJsonPath = true;
        }
    }
    else
    {
        m_PathLength = m_Nodes[m_Depth].head;
        m_Path[m_PathLength] = '\0';

        m_Nodes[m_Depth].validJsonPath = true;
    }

    m_Depth--;
}

bool JsonPath::PopIfKey() NN_NOEXCEPT
{
    if (m_Nodes[m_Depth].type != NodeType_Key)
    {
        return false;
    }
    Pop();

    return true;
}

void JsonPath::InvalidateJsonPath() NN_NOEXCEPT
{
    for (size_t i = m_Nodes[m_Depth].head; i < PathLengthMax; i++)
    {
        m_Path[i] = '.';
    }
    m_Path[PathLengthMax] = '\0';
    m_PathLength = PathLengthMax;

    m_Nodes[m_Depth].validJsonPath = false;
}

}}}}}
