﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskChangePresencePermission.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerUserResource.h>
#include <nn/friends/detail/service/core/friends_UserSettingManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

ChangePresencePermissionTask::ChangePresencePermissionTask() NN_NOEXCEPT :
    m_IsParameterSet(false)
{
}

nn::Result ChangePresencePermissionTask::SetParameter(PresencePermission newValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(newValue == PresencePermission_Self ||
        newValue == PresencePermission_FavoriteFriends ||
        newValue == PresencePermission_Friends,
        ResultInvalidArgument());

    m_NewValue = newValue;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result ChangePresencePermissionTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepChange());

    NN_RESULT_SUCCESS;
}

nn::Result ChangePresencePermissionTask::StepChange() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] ChangePresencePermissionTask::StepChange ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx", WebApi::GetFqdn(), accountId.id);

    char patch[128] = {};
    NN_RESULT_DO(CreateChangePatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));

    NN_RESULT_SUCCESS;
}

nn::Result ChangePresencePermissionTask::CreateChangePatch(char* buffer, size_t size) NN_NOEXCEPT
{
    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteReplaceOperation(writer, "/permissions/presence");
        {
            writer.String(ParameterConverter::ConvertPresencePermission(m_NewValue));
        }
        detail::service::json::JsonPatchGenerator::EndWriteReplaceOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
