﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerProfileResourceList.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPlayLog.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

ProfileResourceListHandler::ProfileResourceListHandler() NN_NOEXCEPT :
    m_OutProfiles(nullptr),
    m_OutProfileExtras(nullptr),
    m_AccountIds(nullptr),
    m_Count(0),
    m_Bits()
{
}

void ProfileResourceListHandler::Initialize(ProfileImpl* outProfiles,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProfiles);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES(count > 0);

    m_OutProfiles = outProfiles;
    m_AccountIds = accountIds;
    m_Count = count;

    std::memset(m_OutProfiles, 0, sizeof (ProfileImpl) * m_Count);
}

void ProfileResourceListHandler::Initialize(ProfileExtraImpl* outProfiles,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProfiles);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES(count > 0);

    m_OutProfileExtras = outProfiles;
    m_AccountIds = accountIds;
    m_Count = count;

    std::memset(m_OutProfileExtras, 0, sizeof (ProfileExtraImpl) * m_Count);
}

void ProfileResourceListHandler::Initialize(ProfileExtraImpl* outProfile) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProfile);

    m_OutProfileExtras = outProfile;
    m_Count = 1;

    std::memset(m_OutProfileExtras, 0, sizeof (ProfileExtraImpl) * m_Count);
}

bool ProfileResourceListHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        std::memset(&m_Profile, 0, sizeof (m_Profile));
        std::memset(&m_Bits, 0, sizeof (m_Bits));
    }

    return true;
}

bool ProfileResourceListHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Bits.accountId && m_Bits.nickname && m_Bits.profileImageUrl)
        {
            m_Profile.data.isValid = true;

            for (int i = 0; i < m_Count; i++)
            {
                if (!m_AccountIds || m_AccountIds[i] == m_Profile.data.accountId)
                {
                    if (m_OutProfiles)
                    {
                        m_OutProfiles[i].data.accountId = m_Profile.data.accountId;
                        m_OutProfiles[i].data.nickname = m_Profile.data.nickname;
                        m_OutProfiles[i].data.profileImageUrl = m_Profile.data.profileImageUrl;
                        m_OutProfiles[i].data.isValid = m_Profile.data.isValid;
                    }
                    if (m_OutProfileExtras)
                    {
                        m_OutProfileExtras[i] = m_Profile;
                    }
                }
            }
        }
    }

    return true;
}

bool ProfileResourceListHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Match("$.items[*].id"))
    {
        if (detail::util::HexToNumber(&m_Profile.data.accountId.id, value))
        {
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.nickname &&
        jsonPath.Match("$.items[*].nickname"))
    {
        nn::util::Strlcpy(m_Profile.data.nickname.name, value, sizeof (m_Profile.data.nickname.name));
        m_Bits.nickname = true;
    }
    else if (!m_Bits.profileImageUrl &&
        jsonPath.Match("$.items[*].thumbnailUrl"))
    {
        nn::util::Strlcpy(m_Profile.data.profileImageUrl.value, value,
            sizeof (m_Profile.data.profileImageUrl.value));
        m_Bits.profileImageUrl = true;
    }
    else if (!m_Bits.playLog &&
        jsonPath.Match("$.items[*].extras.*.playLog"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] PlayLog string is overflowed.\n");
        }
        else if (ParsePlayLog(value, length))
        {
            m_Bits.playLog = true;
        }
    }

    return true;
}

bool ProfileResourceListHandler::ParsePlayLog(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '[' || json[length - 1] != ']')
    {
        return false;
    }

    PlayLogHandler handler;
    handler.Initialize(m_Profile.data.playLog, PlayLogCountMax);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[32];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
