﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerPlayLog.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

PlayLogHandler::PlayLogHandler() NN_NOEXCEPT :
    m_OutPlayLogs(nullptr),
    m_Index(0),
    m_Num(0)
{
}

void PlayLogHandler::Initialize(PlayLog* outPlayLogs, int num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPlayLogs);
    NN_SDK_REQUIRES(num > 0);

    m_OutPlayLogs = outPlayLogs;
    m_Num = num;

    std::memset(m_OutPlayLogs, 0, sizeof (PlayLog) * m_Num);
}

int PlayLogHandler::GetCount() const NN_NOEXCEPT
{
    return m_Index;
}

bool PlayLogHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    std::memset(&m_PlayLog, 0, sizeof (m_PlayLog));
    std::memset(&m_Bits, 0, sizeof (m_Bits));

    return true;
}

bool PlayLogHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (m_Bits.appId && m_Bits.presenceGroupId &&
        m_Bits.totalPlayCount && m_Bits.totalPlayTime &&
        m_Bits.firstPlayedTime && m_Bits.lastPlayedTime)
    {
        if (m_Index < m_Num)
        {
            AddOrMergePlayLog();
        }
    }

    return true;
}

bool PlayLogHandler::OnEndArray(const detail::service::json::JsonPath& jsonPath, size_t numElements) NN_NOEXCEPT
{
    NN_UNUSED(jsonPath);
    NN_UNUSED(numElements);

    if (m_Index > 0)
    {
        // 最後に遊んだ順に並べる。
        std::sort(m_OutPlayLogs, m_OutPlayLogs + m_Index,
            [](const PlayLog& lhs, const PlayLog& rhs)
            {
                return lhs.lastPlayedTime.value > rhs.lastPlayedTime.value;
            });
    }

    return true;
}

bool PlayLogHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.totalPlayCount &&
        jsonPath.Match("$[*].totalPlayCount"))
    {
        m_PlayLog.totalPlayCount = static_cast<int32_t>(value.s);
        m_Bits.totalPlayCount = true;
    }
    else if (!m_Bits.totalPlayTime &&
        jsonPath.Match("$[*].totalPlayTime"))
    {
        m_PlayLog.totalPlayTime = static_cast<int32_t>(value.s);
        m_Bits.totalPlayTime = true;
    }
    else if (!m_Bits.firstPlayedTime &&
        jsonPath.Match("$[*].firstPlayedAt"))
    {
        m_PlayLog.firstPlayedTime.value = value.s;
        m_Bits.firstPlayedTime = true;
    }
    else if (!m_Bits.lastPlayedTime &&
        jsonPath.Match("$[*].lastPlayedAt"))
    {
        m_PlayLog.lastPlayedTime.value = value.s;
        m_Bits.lastPlayedTime = true;
    }

    return true;
}

bool PlayLogHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(isOverflowed);

    if (!m_Bits.appId &&
        jsonPath.Match("$[*].appInfo:appId"))
    {
        if (!detail::util::HexToNumber(&m_PlayLog.appInfo.appId.value, value))
        {
            m_PlayLog.appInfo.appId.value = 0;
        }
        m_Bits.appId = true;
    }
    else if (!m_Bits.presenceGroupId &&
        jsonPath.Match("$[*].appInfo:presenceGroupId"))
    {
        if (!detail::util::HexToNumber(&m_PlayLog.appInfo.presenceGroupId, value))
        {
            m_PlayLog.appInfo.presenceGroupId = 0;
        }
        m_Bits.presenceGroupId = true;
    }

    return true;
}

void PlayLogHandler::AddOrMergePlayLog() NN_NOEXCEPT
{
    for (int i = 0; i < m_Index; i++)
    {
        PlayLog& currentPlayLog = m_OutPlayLogs[i];

        // アプリケーション ID が同一の場合、各項目の評価値の高い方を採用する。
        if (currentPlayLog.appInfo.appId == m_PlayLog.appInfo.appId)
        {
            currentPlayLog.appInfo = m_PlayLog.appInfo;

            // 総プレイ回数は、より多い方。
            if (currentPlayLog.totalPlayCount < m_PlayLog.totalPlayCount)
            {
                currentPlayLog.totalPlayCount = m_PlayLog.totalPlayCount;
            }
            // 総プレイ時間は、より長い方。
            if (currentPlayLog.totalPlayTime < m_PlayLog.totalPlayTime)
            {
                currentPlayLog.totalPlayTime = m_PlayLog.totalPlayTime;
            }
            // 最初に遊んだ時間は、より早い方。
            if (currentPlayLog.firstPlayedTime.value > m_PlayLog.firstPlayedTime.value)
            {
                currentPlayLog.firstPlayedTime = m_PlayLog.firstPlayedTime;
            }
            // 最後に遊んだ時間は、より遅い方。
            if (currentPlayLog.lastPlayedTime.value < m_PlayLog.lastPlayedTime.value)
            {
                currentPlayLog.lastPlayedTime = m_PlayLog.lastPlayedTime;
            }
            return;
        }
    }

    std::memcpy(&m_OutPlayLogs[m_Index], &m_PlayLog, sizeof (m_PlayLog));
    m_Index++;
}

}}}}}
