﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerNotification.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPresence.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

NotificationHandler::NotificationHandler() NN_NOEXCEPT :
    m_AccountId(),
    m_Presence(),
    m_LogoutTime(),
    m_CreationTime(),
    m_RequestType(RequestType_Unknown),
    m_Bits()
{
    m_Type[0] = '\0';
}

const char* NotificationHandler::GetType() const NN_NOEXCEPT
{
    return m_Type;
}

nn::account::NetworkServiceAccountId NotificationHandler::GetAccountId() const NN_NOEXCEPT
{
    return m_AccountId;
}

const FriendPresenceImpl& NotificationHandler::GetFriendPresence() const NN_NOEXCEPT
{
    return m_Presence;
}

RequestType NotificationHandler::GetRequestType() const NN_NOEXCEPT
{
    return m_RequestType;
}

const nn::time::PosixTime NotificationHandler::GetCreationTime() const NN_NOEXCEPT
{
    return m_CreationTime;
}

bool NotificationHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Compare("$"))
    {
        if (!(m_Bits.type && m_Bits.accountId))
        {
            return false;
        }
        if (nn::util::Strncmp(m_Type, "presence_updated", sizeof ("presence_updated")) == 0)
        {
            // オフライン状態の時はログアウト時刻を最終更新時刻扱いにする。
            if (m_Bits.presenceLogoutTime && m_Presence.data.status == PresenceStatus_Offline)
            {
                m_Presence.data.lastUpdateTime = m_LogoutTime;
            }
        }
        else
        {
            std::memset(&m_Presence, 0, sizeof (m_Presence));
        }
    }

    return true;
}

bool NotificationHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.presenceLastUpdateTime &&
        jsonPath.Compare("$.value.updatedAt"))
    {
        m_Presence.data.lastUpdateTime.value = value.s;
        m_Bits.presenceLastUpdateTime = true;
    }
    else if (!m_Bits.presenceLogoutTime &&
        jsonPath.Compare("$.value.logoutAt"))
    {
        m_LogoutTime.value = value.s;
        m_Bits.presenceLogoutTime = true;
    }
    else if (!m_Bits.creationTime &&
        jsonPath.Compare("$.createdAt"))
    {
        m_CreationTime.value = value.s;
        m_Bits.creationTime = true;
    }

    return true;
}

bool NotificationHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(length);

    if (!m_Bits.type &&
        jsonPath.Compare("$.type"))
    {
        nn::util::Strlcpy(m_Type, value, sizeof (m_Type));
        m_Bits.type = true;
    }
    else if (!m_Bits.accountId &&
        jsonPath.Compare("$.value.userId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_AccountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.accountId &&
        jsonPath.Compare("$.value.senderId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_AccountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.presenceStatus &&
        jsonPath.Compare("$.value.state"))
    {
        m_Presence.data.status = ParameterConverter::ConvertPresenceStatus(value);
        m_Bits.presenceStatus = true;
    }
    else if (!m_Bits.presenceAppId &&
        jsonPath.Compare("$.value.extras.friends.appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Presence.data.lastPlayedApp.appId = appId;
            m_Bits.presenceAppId = true;
        }
    }
    else if (!m_Bits.presencePresenceGroupId &&
        jsonPath.Compare("$.value.extras.friends.appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Presence.data.lastPlayedApp.presenceGroupId = presenceGroupId;
            m_Bits.presencePresenceGroupId = true;
        }
    }
    else if (!m_Bits.presenceAppField &&
        jsonPath.Compare("$.value.extras.friends.appField"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The presence string is overflowed.\n");
        }

        m_Bits.presenceAppField = ParseFriendPresenceAppField(value, length);
    }
    else if (!m_Bits.requestType &&
        jsonPath.Compare("$.value.channels[0]")) // 最初に出現したチャンネルだけで十分。
    {
        m_RequestType = ParameterConverter::ConvertRequestType(value);
        m_Bits.requestType = true;
    }

    return true;
}

bool NotificationHandler::ParseFriendPresenceAppField(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '{' || json[length - 1] != '}')
    {
        return false;
    }

    PresenceHandler handler;
    handler.Initialize(m_Presence.data.appField, PresenceAppFieldSize);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[PresenceAppFieldSize];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
