﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/srepo/srepo_StateNotifier.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

UserPresenceManager::UserPresenceManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_IsNpnsConnected(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        m_Records[i].uid = nn::account::InvalidUid;
    }
}

nn::Result UserPresenceManager::DeclareOpenOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultInvalidArgument());

    if (!m_Records[index].isOnlinePlaySessionOpened)
    {
        m_Records[index].isOnlinePlaySessionOpened = true;
        UpdateStatus(index);

        BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[index].uid);
    }

    NN_RESULT_SUCCESS;
}

nn::Result UserPresenceManager::DeclareCloseOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultInvalidArgument());

    if (m_Records[index].isOnlinePlaySessionOpened)
    {
        m_Records[index].isOnlinePlaySessionOpened = false;
        UpdateStatus(index);

        BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[index].uid);
    }

    NN_RESULT_SUCCESS;
}

nn::Result UserPresenceManager::SetPresence(const nn::account::Uid& uid, const UserPresenceImpl& presence, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultInvalidArgument());

    m_Records[index].presence.data.lastPlayedApp = appInfo;

    if (std::memcmp(m_Records[index].presence.data.appField, presence.data.appField, sizeof (m_Records[index].presence.data.appField)) != 0)
    {
        std::memcpy(m_Records[index].presence.data.appField, presence.data.appField, sizeof (m_Records[index].presence.data.appField));
        UpdateLastUpdateTime(index);

        BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[index].uid);
    }
    if (presence.data.onlinePlayDeclaration == OnlinePlayDeclaration_Open)
    {
        DeclareOpenOnlinePlaySession(uid);
    }
    else if (presence.data.onlinePlayDeclaration == OnlinePlayDeclaration_Close)
    {
        DeclareCloseOnlinePlaySession(uid);
    }

    NN_RESULT_SUCCESS;
}

nn::Result UserPresenceManager::GetPresence(UserPresenceViewImpl* outPresence, const nn::account::Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPresence);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultInvalidArgument());

    std::memcpy(outPresence, &m_Records[index].presence, sizeof (UserPresenceViewImpl));

    if (!m_IsNpnsConnected)
    {
        // アプリが設定したプレゼンスデータを見えるようにするために、プレゼンスの状態以外は変更しない。
        outPresence->data.status = PresenceStatus_Offline;
    }

    NN_RESULT_SUCCESS;
}

void UserPresenceManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    std::memset(&m_Records[index], 0, sizeof (Record));
    m_Records[index].uid = uid;
}

void UserPresenceManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].uid = nn::account::InvalidUid;
}

void UserPresenceManager::NotifyUserOpened(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].isOpened = true;
    m_Records[index].isOnlinePlaySessionOpened = false;
    UpdateStatus(index);

    std::memset(m_Records[index].presence.data.appField, 0, sizeof (m_Records[index].presence.data.appField));

    m_Records[index].presence.data.lastPlayedApp = appInfo;

    BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[index].uid);
}

void UserPresenceManager::NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].isOpened = false;
    m_Records[index].isOnlinePlaySessionOpened = false;
    UpdateStatus(index);

    std::memset(m_Records[index].presence.data.appField, 0, sizeof (m_Records[index].presence.data.appField));

    BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[index].uid);
}

void UserPresenceManager::NotifySleep() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // スリープした場合オンラインプレイのセッションが切れることは自明なため、自動的にセッションを閉じる。
    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (!m_Records[i].uid)
        {
            continue;
        }

        if (m_Records[i].isOnlinePlaySessionOpened)
        {
            m_Records[i].isOnlinePlaySessionOpened = false;
            UpdateStatus(i);

            BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[i].uid);
        }
    }
}

void UserPresenceManager::NotifySleepAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_IsNpnsConnected)
    {
        return;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (!m_Records[i].uid)
        {
            continue;
        }

        if (m_Records[i].isOpened)
        {
            // スリープ復帰したのでプレゼンスを再送する。
            BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[i].uid);
        }
    }
}

void UserPresenceManager::NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (isConnected == m_IsNpnsConnected)
    {
        return;
    }

    m_IsNpnsConnected = isConnected;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (!m_Records[i].uid)
        {
            continue;
        }

        if (m_Records[i].isOpened)
        {
            // 表示用の時間を更新する。
            UpdateLastUpdateTime(i);

            if (m_IsNpnsConnected)
            {
                // NPNS サーバーから切断されていた場合、自動的にオフライン状態に遷移している可能性があるため、接続時はプレゼンスを再送する。
                BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(m_Records[i].uid);
            }
        }
    }
}

int UserPresenceManager::FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == uid)
        {
            return i;
        }
    }

    return -1;
}

void UserPresenceManager::UpdateStatus(int index) NN_NOEXCEPT
{
    int32_t prevStatus = m_Records[index].presence.data.status;

    if (m_Records[index].isOpened)
    {
        m_Records[index].presence.data.status =
            m_Records[index].isOnlinePlaySessionOpened ? PresenceStatus_OnlinePlay : PresenceStatus_Online;
    }
    else
    {
        m_Records[index].presence.data.status = PresenceStatus_Offline;
    }

    if (prevStatus != m_Records[index].presence.data.status)
    {
        UpdateLastUpdateTime(index);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        const auto& uid = m_Records[index].uid;
        switch(m_Records[index].presence.data.status)
        {
            case PresenceStatus_Offline:
                nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::Offline);
                break;
            case PresenceStatus_Online:
                nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::Online);
                break;
            case PresenceStatus_OnlinePlay:
                nn::srepo::NotifyFriendPresenceChanged(uid, nn::srepo::FriendPresence::OnlinePlay);
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
#endif
    }
}

void UserPresenceManager::UpdateLastUpdateTime(int index) NN_NOEXCEPT
{
    nn::time::PosixTime updateTime = {};

    if (nn::time::StandardUserSystemClock::GetCurrentTime(&updateTime).IsSuccess())
    {
        m_Records[index].presence.data.lastUpdateTime = updateTime;
    }
    else
    {
        m_Records[index].presence.data.lastUpdateTime.value = 0;
    }
}

}}}}}
