﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_PlayLogManager.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "playlog.bin";
}

PlayLogManager::PlayLogManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid),
    m_Count(0)
{
    std::memset(&m_CurrentApplications, 0, sizeof (m_CurrentApplications));
}

nn::Result PlayLogManager::GetPlayLogList(int* outCount, PlayLog* outPlayLogs, const nn::account::Uid& uid, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outPlayLogs);
    NN_SDK_REQUIRES(count > 0);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outPlayLogs, 0, sizeof (PlayLog) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (count-- == 0)
        {
            break;
        }

        outPlayLogs[actualCount++] = m_Records[i].playLog;
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result PlayLogManager::HasPlayLog(bool* out, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    *out = (m_Count > 0);

    NN_RESULT_SUCCESS;
}

nn::Result PlayLogManager::RemoveUploadedPlayLog(const nn::account::Uid& uid, const PlayLog* playLogs, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(playLogs);

    if (count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    bool isDirty = false;

    for (int i = 0; i < count; i++)
    {
        int c = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, playLogs[i],
            [](const Record& lhs, const PlayLog& rhs)
            {
                return std::memcmp(&lhs.playLog, &rhs, sizeof (PlayLog)) == 0;
            });

        if (c < m_Count)
        {
            m_Count = c;
            isDirty = true;
        }
    }

    if (isDirty)
    {
        NN_RESULT_DO(Save());
    }

    NN_RESULT_SUCCESS;
}

nn::Result PlayLogManager::Clear(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    m_Count = 0;

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

void PlayLogManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

void PlayLogManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_CurrentApplications); i++)
    {
        if (m_CurrentApplications[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    std::memset(&m_CurrentApplications[index], 0, sizeof (CurrentApplicationInfo));
    m_CurrentApplications[index].uid = uid;
}

void PlayLogManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchUser(uid);

    if (index != -1)
    {
        m_CurrentApplications[index].uid = nn::account::InvalidUid;
    }
}

void PlayLogManager::NotifyUserOpened(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    nn::Result result = Update(uid, appInfo);

    if (result.IsFailure())
    {
        NN_DETAIL_FRIENDS_INFO("[friends] PlayLogManager::Update failed. code = %03d-%04d\n",
            result.GetModule(), result.GetDescription());
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchUser(uid);

    if (index != -1)
    {
        m_CurrentApplications[index].appInfo = appInfo;
    }
}

void PlayLogManager::NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchUser(uid);

    if (index != -1)
    {
        nn::Result result = Update(uid, m_CurrentApplications[index].appInfo);

        if (result.IsFailure())
        {
            NN_DETAIL_FRIENDS_INFO("[friends] PlayLogManager::Update failed. code = %03d-%04d\n",
                result.GetModule(), result.GetDescription());
        }

        m_CurrentApplications[index].appInfo.appId = nn::ApplicationId::GetInvalidId();
    }
}

void PlayLogManager::NotifySleep() NN_NOEXCEPT
{
}

void PlayLogManager::NotifySleepAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < NN_ARRAY_SIZE(m_CurrentApplications); i++)
    {
        if (m_CurrentApplications[i].uid && m_CurrentApplications[i].appInfo.appId != nn::ApplicationId::GetInvalidId())
        {
            nn::Result result = Update(m_CurrentApplications[i].uid, m_CurrentApplications[i].appInfo);

            if (result.IsFailure())
            {
                NN_DETAIL_FRIENDS_INFO("[friends] PlayLogManager::Update failed. code = %03d-%04d\n",
                    result.GetModule(), result.GetDescription());
            }
        }
    }
}

nn::Result PlayLogManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result PlayLogManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > PlayLogCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > PlayLogCountMax)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));

    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result PlayLogManager::Save() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    // 新規にデータを作成する際、ストレージの容量不足にならないよう、不要なファイルを削除する。
    NN_RESULT_DO(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    bool isDeleted = true;

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            isDeleted = false;
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        if (isDeleted)
        {
            NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());
        }

        NN_RESULT_SUCCESS;
    }

    {
        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

int PlayLogManager::SearchUser(const nn::account::Uid& uid) NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_CurrentApplications); i++)
    {
        if (uid == m_CurrentApplications[i].uid)
        {
            return i;
        }
    }

    return -1;
}

nn::Result PlayLogManager::Update(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    nn::ncm::ApplicationId appId = {appInfo.appId.value};

    auto statistics = nn::pdm::QueryPlayStatistics(appId, uid);

    if (!statistics)
    {
        NN_DETAIL_FRIENDS_INFO("[friends] The play statistics is not found.\n");
        NN_RESULT_THROW(ResultApplicationInfoNotRegistered());
    }

    Record record = {};

    record.playLog.appInfo = appInfo;
    record.playLog.totalPlayCount = statistics->totalPlayCount;
    record.playLog.totalPlayTime = statistics->totalPlayTime;
    record.playLog.firstPlayedTime = nn::time::InputPosixTimeMin;
    record.playLog.firstPlayedTime.value += static_cast<int64_t>(statistics->firstEventTime.userClockTime) * 60ll;
    record.playLog.lastPlayedTime = nn::time::InputPosixTimeMin;
    record.playLog.lastPlayedTime.value += static_cast<int64_t>(statistics->latestEventTime.userClockTime) * 60ll;

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    Load(uid);

    int index = -1;

    for (int i = 0; i < m_Count; i++)
    {
        if (appInfo.appId == m_Records[i].playLog.appInfo.appId)
        {
            index = i;
            break;
        }
    }

    if (index != -1 && std::memcmp(&record, &m_Records[index], sizeof (Record)) == 0)
    {
        NN_DETAIL_FRIENDS_INFO("[friends] There is no change.\n");
        NN_RESULT_SUCCESS;
    }

    m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, record,
        [](const Record& lhs, const Record& rhs)
        {
            return lhs.playLog.appInfo.appId == rhs.playLog.appInfo.appId;
        });

    m_Count = detail::service::util::ArrayAccessor::InsertToTopEntry(m_Records, m_Count, record);

    NN_RESULT_TRY(Save())
        NN_RESULT_CATCH(ResultNetworkServiceAccountNotLinked)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

}}}}}
