﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_OverlayNotificationSender.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/format/ovln_FriendsMessage.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    // プレゼンス更新通知をある程度保持できるようにするため、キューの長さを多めにしておく。
    const int OverlayNotificationQueueLength = 16;
}

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    nn::ovln::SenderForOverlayType g_Sender;
    bool g_IsInitialized = false;
}

namespace
{
    void Initialize() NN_NOEXCEPT
    {
        if (!g_IsInitialized)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderWithQueue(&g_Sender, OverlayNotificationQueueLength));
            g_IsInitialized = true;
        }
    }
}

void OverlayNotificationSender::SendListSummaryMessage(const nn::account::Uid& uid, int32_t onlineFriendCount,
    int32_t unreadFriendRequestCount, int32_t readFriendRequestCount) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    Initialize();

    nn::ovln::Message message = {};
    nn::ovln::format::ListSummary format = {};

    message.tag = nn::ovln::format::ListSummaryTag;

    format.uid = uid;
    format.onlineFriendCount = onlineFriendCount;
    format.unreadFriendRequestCount = unreadFriendRequestCount;
    format.readFriendRequestCount = readFriendRequestCount;

    message.dataSize = sizeof (format);
    std::memcpy(&message.data, &format, sizeof (format));

    nn::ovln::Send(&g_Sender, message);

    NN_DETAIL_FRIENDS_INFO("[friends] OverlayNotificationSender::SendListSummaryMessage(%016llx_%016llx, %d, %d, %d).\n",
        uid._data[0], uid._data[1], onlineFriendCount, unreadFriendRequestCount, readFriendRequestCount);
}

void OverlayNotificationSender::SendFriendOnlineMessage(const nn::account::Uid& uid,
    nn::account::NetworkServiceAccountId friendAccountId, int32_t elapsedSecondsFromLastUpdate) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    Initialize();

    nn::ovln::Message message = {};
    nn::ovln::format::FriendOnline format = {};

    message.tag = nn::ovln::format::FriendOnlineTag;

    format.uid = uid;
    format.friendAccountId = friendAccountId;
    format.elapsedSecondsFromLastUpdate = elapsedSecondsFromLastUpdate;

    message.dataSize = sizeof (format);
    std::memcpy(&message.data, &format, sizeof (format));

    nn::ovln::Send(&g_Sender, message);

    NN_DETAIL_FRIENDS_INFO("[friends] OverlayNotificationSender::SendFriendOnlineMessage(%016llx_%016llx, %016llx, %d)\n",
        uid._data[0], uid._data[1], friendAccountId.id, elapsedSecondsFromLastUpdate);
}

void OverlayNotificationSender::SendFriendRequestReceivedMessage(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    Initialize();

    nn::ovln::Message message = {};
    nn::ovln::format::FriendRequestReceived format = {};

    message.tag = nn::ovln::format::FriendRequestReceivedTag;

    format.uid = uid;

    message.dataSize = sizeof (format);
    std::memcpy(&message.data, &format, sizeof (format));

    nn::ovln::Send(&g_Sender, message);

    NN_DETAIL_FRIENDS_INFO("[friends] OverlayNotificationSender::SendFriendRequestReceivedMessage(%016llx_%016llx)\n",
        uid._data[0], uid._data[1]);
}

}}}}}
