﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_NotificationReceiver.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>
#include <nn/friends/detail/service/core/friends_OverlayNotificationSender.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerNotification.h>
#include <nn/friends/friends_ApiNotificationSettings.h>
#include <nn/npns/npns_Api.h>
#include <nn/npns/npns_ApiSystem.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const nn::ApplicationId ProcessAppId = {0x010000000000000E};
}

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[32 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    void NotifyIfPresenceUpdated(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
        const FriendPresenceImpl& prevPresence, const FriendPresenceImpl& nextPresence, nn::time::PosixTime creationTime) NN_NOEXCEPT
    {
        FriendPresenceManager::GetInstance().UpdatePresence(uid, accountId, nextPresence, creationTime);

        // 前回のプレゼンス更新時刻が取得できないならオーバーレイ通知は送信しない。
        if (prevPresence.data.lastUpdateTime.value == 0)
        {
            return;
        }

        // ユーザーアカウントが Open されていてプレゼンスがオフラインからオンラインに変更された時、オーバーレイ通知を送る。
        if (Account::IsUserOpened(uid) && prevPresence.data.status == PresenceStatus_Offline &&
            (nextPresence.data.status == PresenceStatus_Online || nextPresence.data.status == PresenceStatus_OnlinePlay))
        {
            nn::settings::system::AccountNotificationSettings notificationSettings = {};
            bool isSendable = false;

            // 3.0.0 NUP 時点では、FriendSetting::onlineNotification フラグを見ない代わりに、アカウントの通知設定を見る。
            if (GetNotificationSettings(&notificationSettings, uid).IsSuccess())
            {
                FriendSettingImpl setting = {};

                if (FriendListManager::GetInstance().GetFriendSetting(&setting, uid, accountId).IsSuccess())
                {
                    switch (notificationSettings.friendPresenceOverlayPermission)
                    {
                    case nn::settings::system::FriendPresenceOverlayPermission_NoDisplay:
                        {
                        }
                        break;
                    case nn::settings::system::FriendPresenceOverlayPermission_FavoriteFriends:
                        {
                            if (setting.data.favorite)
                            {
                                isSendable = true;
                            }
                        }
                        break;
                    case nn::settings::system::FriendPresenceOverlayPermission_Friends:
                        {
                            isSendable = true;
                        }
                    default:
                        break;
                    }
                }
            }

            if (isSendable)
            {
                nn::time::PosixTime now = {};

                if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&now).IsSuccess())
                {
                    OverlayNotificationSender::SendFriendOnlineMessage(uid, accountId,
                        static_cast<int32_t>((now - prevPresence.data.lastUpdateTime).GetSeconds()));
                }
            }
        }
    }

    void NotifyFriendRequestReceived(const nn::account::Uid& uid, RequestType requestType) NN_NOEXCEPT
    {
        BackgroundTaskManager::GetInstance().NotifyFriendRequestReceived(uid);

        // 対面フレンド申請はお互いがフレンド申請していることを知っているので、わざわざオーバーレイで通知しない。
        if (Account::IsUserOpened(uid) && requestType != RequestType_Faced)
        {
            OverlayNotificationSender::SendFriendRequestReceivedMessage(uid);
        }
    }

    bool AnalyzeNotificationData(const nn::npns::NotificationData& data) NN_NOEXCEPT
    {
        if (data.GetPayloadSize() == 0)
        {
            return false;
        }

        detail::service::core::NotificationHandler handler;

        detail::service::json::JsonMemoryInputStream stream;
        detail::service::util::Cancelable cancelable;

        char stringBuffer[1024];
        char streamBuffer[32];

        stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
        stream.Open(data.GetPayload(), data.GetPayloadSize());

        if (detail::service::json::Parse(handler, stream, cancelable).IsFailure())
        {
            return false;
        }

        NN_DETAIL_FRIENDS_INFO("[friends] Notification('%s', %016llx_%016llx) was received.\n",
            handler.GetType(), data.GetUid()._data[0], data.GetUid()._data[1]);

        if (nn::util::Strncmp(handler.GetType(), "presence_updated", sizeof ("presence_updated")) == 0)
        {
            FriendPresenceImpl prevPresence = {};
            FriendPresenceManager::GetInstance().GetPresence(&prevPresence, data.GetUid(), handler.GetAccountId());

            NotifyIfPresenceUpdated(data.GetUid(), handler.GetAccountId(), prevPresence,
                handler.GetFriendPresence(), handler.GetCreationTime());
        }
        else if (nn::util::Strncmp(handler.GetType(), "friend_request_received", sizeof ("friend_request_received")) == 0)
        {
            NotifyFriendRequestReceived(data.GetUid(), handler.GetRequestType());
        }
        else if (nn::util::Strncmp(handler.GetType(), "friend_request_authorized", sizeof ("friend_request_authorized")) == 0)
        {
            BackgroundTaskManager::GetInstance().NotifyFriendRequestAccepted(data.GetUid());
        }
        else if (nn::util::Strncmp(handler.GetType(), "friend_deleted", sizeof ("friend_deleted")) == 0)
        {
            BackgroundTaskManager::GetInstance().NotifyFriendDeleted(data.GetUid());
        }
        else
        {
            NN_DETAIL_FRIENDS_INFO("[friends] The unhandled type payload was received.\n");
        }

        return true;
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::ListenTo(ProcessAppId));

        nn::os::SystemEvent receiveEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::GetReceiveEvent(receiveEvent));

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitAny(g_StopEvent.GetBase(), receiveEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                break;
            }

            receiveEvent.Clear();

            while (NN_STATIC_CONDITION(true))
            {
                nn::npns::NotificationData data;

                nn::Result result = nn::npns::Receive(&data);

                if (nn::npns::ResultNotReceived::Includes(result))
                {
                    break;
                }
                if (result.IsFailure())
                {
                    NN_DETAIL_FRIENDS_INFO("[friends] nn::npns::Receive() failed. code = %03d-%04d\n",
                        result.GetModule(), result.GetDescription());
                    break;
                }

                if (!AnalyzeNotificationData(data))
                {
                    NN_DETAIL_FRIENDS_INFO("[friends] The invalid format payload was received.\n");
                }
            }
        }
    }
}

void NotificationReceiver::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(friends, NotificationReceiver)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(friends, NotificationReceiver));
    nn::os::StartThread(&g_Thread);
}

void NotificationReceiver::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}

}}}}}
