﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    inline bool IsZeroMemory(const Bit64* mem, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES((size % 8) == 0);

        int count = static_cast<int>(size) / 8;

        for (int i = 0; i < count; i++)
        {
            if (mem[i])
            {
                return false;
            }
        }

        return true;
    }
}

FriendPresenceManager::FriendPresenceManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_IsNetworkConnected(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        m_Records[i].uid = nn::account::InvalidUid;
    }
}

void FriendPresenceManager::GetPresence(FriendPresenceImpl* outPresence,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPresence);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        std::memset(outPresence, 0, sizeof (FriendPresenceImpl));
        return;
    }

    for (int i = 0; i < m_Records[index].count; i++)
    {
        if (m_Records[index].accountIds[i] == accountId)
        {
            if (m_Records[index].presences[i].data.status == PresenceStatus_Offline)
            {
                std::memset(outPresence, 0, sizeof (FriendPresenceImpl));

                outPresence->data.status = PresenceStatus_Offline;
                outPresence->data.lastUpdateTime = m_Records[index].presences[i].data.lastUpdateTime;

                // 更新時刻が 0 の時、相手に情報を公開していない。
                if (outPresence->data.lastUpdateTime.value > 0)
                {
                    outPresence->data.lastPlayedApp = m_Records[index].presences[i].data.lastPlayedApp;
                }
            }
            else
            {
                std::memcpy(outPresence, &m_Records[index].presences[i], sizeof (FriendPresenceImpl));
            }
            return;
        }
    }
}

void FriendPresenceManager::GetCount(int* outCount, const nn::account::Uid& uid, PresenceStatusFilter statusFilter) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        *outCount = 0;
        return;
    }

    int count = 0;

    for (int i = 0; i < m_Records[index].count; i++)
    {
        PresenceStatus status = static_cast<PresenceStatus>(m_Records[index].presences[i].data.status);

        switch (statusFilter)
        {
        case PresenceStatusFilter_None:
            {
                count++;
            }
            break;
        case PresenceStatusFilter_Online:
            {
                if (status == PresenceStatus_Online)
                {
                    count++;
                }
            }
            break;
        case PresenceStatusFilter_OnlinePlay:
            {
                if (status == PresenceStatus_OnlinePlay)
                {
                    count++;
                }
            }
            break;
        case PresenceStatusFilter_OnlineOrOnlinePlay:
            {
                if (status == PresenceStatus_Online || status == PresenceStatus_OnlinePlay)
                {
                    count++;
                }
            }
            break;
        default:
            break;
        }
    }

    *outCount = count;
}

void FriendPresenceManager::UpdateFriendList(const nn::account::Uid& uid, const FriendResource* resources, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(resources);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    if (count > FriendCountMax)
    {
        count = FriendCountMax;
    }

    bool isUpdated = false;

    if (count != m_Records[index].count)
    {
        isUpdated = true;
    }

    if (!isUpdated)
    {
        for (int i = 0; i < count; i++)
        {
            int friendIndex = FindFriend(index, resources[i].accountId);

            if (friendIndex == -1)
            {
                isUpdated = true;
                break;
            }
            if (m_Records[index].presences[friendIndex] != resources[i].presence)
            {
                isUpdated = true;
                break;
            }
        }
    }

    // ユーザー数変更確認＋新規追加確認を行っているため、既存ユーザーの削除確認は行わなくてもよい。
    // ユーザー数に変化がなく新規追加が発生している＝誰か削除されているとなるため。

    const FriendPresenceImpl OfflinePresence = {};

    for (int i = 0; i < count; i++)
    {
        m_Records[index].accountIds[i] = resources[i].accountId;
        m_Records[index].presences[i] = m_IsNetworkConnected ? resources[i].presence : OfflinePresence;
    }

    m_Records[index].count = count;

    if (m_IsNetworkConnected && isUpdated)
    {
        NotificationEventHandler::GetInstance().NotifyFriendListUpdated(m_Records[index].uid);
    }
}

void FriendPresenceManager::UpdatePresence(const nn::account::Uid& uid, const FriendResource& resource) NN_NOEXCEPT
{
    nn::time::PosixTime creationTime = {};

    UpdatePresence(uid, resource.accountId, resource.presence, creationTime);
}

void FriendPresenceManager::UpdatePresence(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
    const FriendPresenceImpl& presence, nn::time::PosixTime creationTime) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_IsNetworkConnected)
    {
        return;
    }

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    for (int i = 0; i < m_Records[index].count; i++)
    {
        if (m_Records[index].accountIds[i] == accountId)
        {
            bool isUpdatable = false;

            if (creationTime.value > 0)
            {
                if (m_Records[index].presences[i].data.lastUpdateTime < creationTime)
                {
                    isUpdatable = true;
                }
            }
            else
            {
                if (m_Records[index].presences[i].data.lastUpdateTime < presence.data.lastUpdateTime)
                {
                    isUpdatable = true;
                }
            }

            if (isUpdatable)
            {
                m_Records[index].presences[i] = presence;

                NotificationEventHandler::GetInstance().NotifyFriendPresenceUpdated(m_Records[index].uid, m_Records[index].accountIds[i]);
                return;
            }
        }
    }
}

void FriendPresenceManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    std::memset(&m_Records[index], 0, sizeof (Record));
    m_Records[index].uid = uid;
}

void FriendPresenceManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].uid = nn::account::InvalidUid;
}

void FriendPresenceManager::NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (isConnected == m_IsNetworkConnected)
    {
        return;
    }

    if (!m_IsNetworkConnected)
    {
        m_IsNetworkConnected = true;
    }
    else
    {
        for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
        {
            if (m_Records[i].uid != nn::account::InvalidUid)
            {
                if (!IsZeroMemory(reinterpret_cast<const Bit64*>(m_Records[i].presences), sizeof (m_Records[i].presences)))
                {
                    std::memset(m_Records[i].presences, 0, sizeof (m_Records[i].presences));

                    NotificationEventHandler::GetInstance().NotifyFriendListUpdated(m_Records[i].uid);
                }
            }
        }

        m_IsNetworkConnected = false;
    }
}

int FriendPresenceManager::FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == uid)
        {
            return i;
        }
    }

    return -1;
}

int FriendPresenceManager::FindFriend(int userIndex, nn::account::NetworkServiceAccountId accountId) const NN_NOEXCEPT
{
    for (int i = 0; i < m_Records[userIndex].count; i++)
    {
        if (m_Records[userIndex].accountIds[i] == accountId)
        {
            return i;
        }
    }

    return -1;
}

}}}}}
