﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/util/friends_LanguageCode.h>
#include <nn/friends/friends_Result.h>
#include <nn/account.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace friends { namespace detail {

namespace
{
    bool g_IsCheckUserStatusEnabled = true;
}

nn::account::Uid GetCurrentUser() NN_NOEXCEPT
{
    nn::account::Uid uid = nn::account::InvalidUid;
    int length = 0;

    nn::Result result = nn::account::ListOpenUsers(&length, &uid, 1);

    if (result.IsFailure())
    {
        NN_SDK_ASSERT(false, "nn::account::ListOpenUsers failed.");
    }

    NN_SDK_ASSERT_EQUAL(length, 1);

    return uid;
}

nn::Result CheckUserStatus(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidArgument());

    if (!g_IsCheckUserStatusEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    nn::account::Uid uids[nn::account::UserCountMax] = {};
    int count = 0;

    NN_RESULT_DO(nn::account::ListOpenUsers(&count, uids, NN_ARRAY_SIZE(uids)));

    for (int i = 0; i < count; i++)
    {
        if (uids[i] == uid)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultUserNotOpened());
}

void EnableCheckUserStatus(bool isEnabled) NN_NOEXCEPT
{
    g_IsCheckUserStatusEnabled = isEnabled;
}

nn::Result CheckInternetRequestAccepted() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()),
        ResultInternetRequestNotAccepted());

    NN_RESULT_SUCCESS;
}

nn::Result NormalizeInAppScreenName(InAppScreenName* out, const InAppScreenName& in) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    size_t inLength = static_cast<size_t>(nn::util::Strnlen(in.name, sizeof (in.name)));

    if (inLength > 0)
    {
        NN_RESULT_THROW_UNLESS(nn::util::CopyUtf8String(out->name, sizeof (out->name),
            in.name, inLength, InAppScreenNameLengthMax), ResultInvalidArgument());
    }

    NN_RESULT_THROW_UNLESS(detail::util::VerifyLanguageCode(in.language), ResultInvalidArgument());
    nn::util::Strlcpy(out->language.string, in.language.string, sizeof (in.language.string));

    NN_RESULT_SUCCESS;
}

nn::Result GetMiiImageUrl(Url* outUrl, const MiiImageUrlParam& miiImageUrlParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outUrl);

    if (miiImageUrlParam.value[0])
    {
        nn::util::SNPrintf(outUrl->value, sizeof (outUrl->value),
            "https://mii-secure-%%.cdn.nintendo.net/%s_normal_face.png", miiImageUrlParam.value);
    }
    else
    {
        outUrl->value[0] = '\0';
    }

    NN_RESULT_SUCCESS;
}

}}}
