﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include "fgm_Resource.h"
#include "fgm_Debugger.h"

namespace nn     {
namespace fgm    {
namespace server {

nn::os::Mutex g_ResourceListLock(false);
nn::util::IntrusiveList<Resource, nn::util::IntrusiveListBaseNodeTraits<Resource>> g_ResourceList;

nn::Result Resource::SetOperatingPoint()
NN_NOEXCEPT
{
    nn::Result result;

    Setting min = RequestManager::Min();
    Setting max = RequestManager::Max();

    if ((result = Set(min, max)).IsSuccess())
    {
        #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        Setting current;
        if ((Get(&current)).IsSuccess())
        {
            DebugEntry entry;
            entry.id                  = m_Id;
            entry.event               = EventId_Completion;
            entry.u.completion.min    = min;
            entry.u.completion.max    = max;
            entry.u.completion.actual = current;
            LogEntry(&entry);
        }
        #endif
        result = RequestManager::Signal();
    }

    return result;
}

nn::Result Resource::GetOperatingPoint(Setting* pCurrentSettingOut)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ResourceLock);
    return Get(pCurrentSettingOut);
}

nn::Result Resource::AddRequest(Request* pRequestIn)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ResourceLock);
    return RequestManager::Add(pRequestIn);
}

nn::Result Resource::UpdateRequest(Request* pRequestIn, Setting newMin, Setting newMax)
NN_NOEXCEPT
{
    nn::Result result;

    #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    DebugEntry entry;
    entry.id                 = m_Id;
    entry.event              = EventId_Set;
    entry.u.request.pid      = pRequestIn->GetPid();
    entry.u.request.min      = newMin;
    entry.u.request.max      = newMax;
    entry.u.request.priority = pRequestIn->GetPriority();
    LogEntry(&entry);
    #endif

    std::lock_guard<nn::os::Mutex> lock(m_ResourceLock);
    if ((result = RequestManager::Update(pRequestIn, newMin, newMax)).IsSuccess())
    {
        result = SetOperatingPoint();
    }
    return result;
}

nn::Result Resource::DeleteRequest(Request* pRequestIn)
NN_NOEXCEPT
{
    nn::Result result;

    #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    DebugEntry entry;
    entry.id                 = m_Id;
    entry.event              = EventId_Unset;
    entry.u.request.pid      = pRequestIn->GetPid();
    entry.u.request.min      = pRequestIn->GetMin();
    entry.u.request.max      = pRequestIn->GetMax();
    entry.u.request.priority = pRequestIn->GetPriority();
    LogEntry(&entry);
    #endif

    std::lock_guard<nn::os::Mutex> lock(m_ResourceLock);
    if ((result = RequestManager::Delete(pRequestIn)).IsSuccess())
    {
        result = SetOperatingPoint();
    }
    return result;
}

nn::Result Resource::Initialize(Setting min, Setting max)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ResourceListLock);

    for (auto resource  = g_ResourceList.begin();
              resource != g_ResourceList.end();
              resource++)
    {
        // check for duplicates
        NN_ABORT_UNLESS(resource->GetId() != m_Id);
    }

    RequestManager::SetRange(min, max);

    g_ResourceList.push_back(*this);

    return ResultSuccess();
}

nn::Result Resource::Finalize()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> resource_lock(m_ResourceLock);
    std::lock_guard<nn::os::Mutex> list_lock(g_ResourceListLock);
    g_ResourceList.erase(g_ResourceList.iterator_to(*this));
    return ResultSuccess();
}

Resource::Resource(Module id)
NN_NOEXCEPT :
    m_ResourceLock(false),
    m_Id(id)
{

}

}}}
