﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <nn/fgm/sfdl/fgm.sfdl.h>
#include <nn/fgm/fgm_Types.h>

#include "fgm_Request.h"
#include "fgm_Resource.h"

namespace nn     {
namespace fgm    {
namespace server {

nn::psc::PmState Request::m_PmState = nn::psc::PmState_FullAwake;

nn::Result Request::Initialize(nn::sf::Out<nn::sf::NativeHandle> handleOut, Module id, nn::Bit64 pid)
NN_NOEXCEPT
{
    nn::Result result = nn::fgm::ResultModuleNotFound();

    if (m_Resource != nullptr)
    {
        return nn::fgm::ResultAlreadyInitialized();
    }

    std::lock_guard<nn::os::Mutex> lock(g_ResourceListLock);

    for (auto resource  = g_ResourceList.begin();
              resource != g_ResourceList.end();
              resource++)
    {
        if (resource->GetId() == id)
        {
            std::lock_guard<nn::os::Mutex> lock(m_Lock);

            m_Resource  = &(*resource);
            m_Id        = id;
            m_Min       = Setting_Min;
            m_Max       = Setting_Max;
            m_Pid.value = pid;

            if ((result = m_Resource->AddRequest(this)).IsSuccess())
            {
                handleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));
            }
        }
    }

    return result;
}

nn::Result Request::Set(Setting min, Setting max)
NN_NOEXCEPT
{
    if (m_Resource == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if (m_PmState == nn::psc::PmState_SleepReady)
    {
        return nn::fgm::ResultNotOperational();
    }

    if (min > max)
    {
        return nn::fgm::ResultInvalidRange();
    }

    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return m_Resource->UpdateRequest(this, min, max);
}

nn::Result Request::Get(nn::sf::Out<uint32_t> pSettingOut)
NN_NOEXCEPT
{
    nn::Result result;
    Setting    current;

    if (m_Resource == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if (m_PmState == nn::psc::PmState_SleepReady)
    {
        return nn::fgm::ResultNotOperational();
    }

    if ((result = m_Resource->GetOperatingPoint(&current)).IsSuccess())
    {
        *pSettingOut = current;
    }

    return result;
}

nn::Result Request::Signal()
NN_NOEXCEPT
{
    m_SystemEvent.Signal();
    return ResultSuccess();
}

nn::Result Request::Cancel()
NN_NOEXCEPT
{
    if (m_Resource == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    m_SystemEvent.Signal();
    return ResultSuccess();
}

Request::Request(Priority priority)
NN_NOEXCEPT :
    m_Resource(nullptr),
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
    m_Priority(priority),
    m_Lock(false)
{

}

Request::~Request()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if (m_Resource != nullptr)
    {
        m_Resource->DeleteRequest(this);
        m_Resource = nullptr;
    }
}

}}}
