﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pcv/pcv.h>
#include <nn/fgm/detail/fgm_Log.h>
#include <nn/fgm/fgm_Types.h>

#include "../fgm_Resource.h"

namespace nn     {
namespace fgm    {
namespace server {
namespace detail {

class NXPCVResource : private Resource
{
private:
    nn::pcv::Module  m_Id;
    nn::pcv::ClockHz m_Rates[nn::pcv::MaxNumClockRates];
    int              m_NumRates;
    Setting          m_LastSet;
    bool             m_DisableAllowed;

public:
    nn::Result Set(Setting min, Setting max)
    NN_NOEXCEPT
    {
        // SIGLO-66061: workaround
        // If GetPossibleClockRates fails, FGM skips Set.
        if (m_NumRates == 0)
        {
            return ResultSuccess();
        }

        nn::Result result;

        if ((m_Id == nn::pcv::Module_Cpu)
            || (m_Id == nn::pcv::Module_Gpu)
            || (m_Id == nn::pcv::Module_Emc))
        {
            nn::pcv::ClockHz rate = 0;
            NN_RESULT_DO(nn::pcv::GetClockRate(&rate, m_Id));
            if (min == rate)
            {
                return ResultSuccess();
            }
        }
        else
        {
            if (min == m_LastSet)
            {
                return ResultSuccess();
            }
        }

        // PCV auto-enables clock on the first non-0 request,
        // but it does not list 0 as valid rate, nor does it
        // auto turn off the clock when it is requested to set it to 0.
        // If we get a request to set clock to Setting_Min,
        // explictly disable the clock.

        if (m_DisableAllowed && (min == Setting_Min))
        {
            result = nn::pcv::SetClockEnabled(m_Id, false);
        }
        else if (m_DisableAllowed && (m_LastSet == Setting_Min))
        {
            result = nn::pcv::SetClockEnabled(m_Id, true);
            if (result.IsSuccess())
            {
                result = nn::pcv::SetClockRate(m_Id, min);
                if (!result.IsSuccess())
                {
                    // attempt to resetore clock state
                    nn::pcv::SetClockEnabled(m_Id, false);
                }
            }
        }
        else
        {
            result = nn::pcv::SetClockRate(m_Id, min);
        }

        if (result.IsSuccess())
        {
            m_LastSet = min;
        }

        return result;
    }

    nn::Result Get(Setting* current)
    NN_NOEXCEPT
    {
        // SIGLO-66061: workaround
        // If GetPossibleClockRates fails, FGM skips Get.
        if (m_NumRates == 0)
        {
            return ResultSuccess();
        }

        nn::Result result;
        nn::pcv::ModuleState state;

        if ((result = nn::pcv::GetState(&state, m_Id)).IsSuccess())
        {
            *current = state.clockFrequency;
        }

        return result;
    }

    NXPCVResource(nn::fgm::Module fgmId, nn::pcv::Module pcvId, bool disableAllowed)
    NN_NOEXCEPT :
          Resource(fgmId),
          m_Id(pcvId),
          m_NumRates(0),
          m_LastSet(Setting_Min),
          m_DisableAllowed(disableAllowed)
    {

    }

    nn::Result Initialize()
    {
        nn::Result result;
        Setting    min = Setting_Max;
        Setting    max = Setting_Min;

        nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;

        NN_DETAIL_FGM_WARN_UNLESS_RESULT_SUCCESS(result = nn::pcv::GetPossibleClockRates(
            &listType, m_Rates, &m_NumRates, m_Id, sizeof(m_Rates) / sizeof(m_Rates[0])));

        if (result.IsFailure())
        {
            // SIGLO-66061: workaround
            // If GetPossibleClockRates fails, NXPCVResource keeps invalid value and alive.
            NN_DETAIL_FGM_WARN("Failed to initialize id:%d.\n", Resource::GetId());
            m_NumRates = 0;
            min = max = Setting_Min;
        }
        else
        {
            NN_ABORT_UNLESS(listType == nn::pcv::ClockRatesListType_Discrete);
            NN_ABORT_UNLESS(m_NumRates > 0);

            for (int i = 0; i < m_NumRates; i++)
            {
                if (m_Rates[i] < min) min = m_Rates[i];
                if (m_Rates[i] > max) max = m_Rates[i];
            }

            // Leave it at Setting_Min for modules that can stop clock
            if (!m_DisableAllowed)
            {
                Get(&m_LastSet);
            }
        }

        // If clock can be turned off, include
        // Setting_Min in the allowed range.
        return Resource::Initialize(m_DisableAllowed ? Setting_Min : min, max);
    }
};

}}}}
