﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fatalsrv/fatalsrv_CpuContext.h>
#include <nn/util/util_ScopeExit.h>

#include "fatal_CpuContext.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Types.common.h>
#include <nn/os/os_Config.h>

#include <nn/fatal/detail/fatal_Log.h>

extern "C" void _start() __attribute__((weak));;

namespace nn { namespace fatal { namespace detail {
    namespace
    {
        template <typename T>
        T GetAddressValueWithQuery(uintptr_t address)
        {
            svc::MemoryInfo block;
            svc::PageInfo   page;

            nn::Result result = svc::QueryMemory(&block, &page, address);
            if (result.IsFailure())
            {
                return 0;
            }
            if (!(block.permission & svc::MemoryPermission_Read))
            {
                return 0;
            }

            return *reinterpret_cast<T*>(address);
        }

        template <typename PtrType, int InvalidAddressPattern>
        void FillCallstack(uintptr_t* out, int* outCount, int maxCount, uintptr_t frameAddress, uintptr_t linkAddress) NN_NOEXCEPT
        {
            *outCount = 0;
            for (int i = 0; i < maxCount; i++)
            {
                if (!frameAddress)
                {
                    break;
                }
                if (frameAddress & InvalidAddressPattern)
                {
                    break;
                }
                struct
                {
                    PtrType frame;
                    PtrType lr;
                } frame;
                {
                    frame.frame = GetAddressValueWithQuery<PtrType>(frameAddress);
                    frame.lr = GetAddressValueWithQuery<PtrType>(frameAddress + sizeof(frame.frame));

                    if (frame.frame == 0 || frame.lr == 0)
                    {
                        // 不正なアドレスをたどってしまった場合は中断する
                        break;
                    }
                }
                linkAddress = frame.lr;
                frameAddress = frame.frame;
                out[i] = linkAddress;
                *outCount = i + 1;
            }
        }

        typedef void(*FillBacktraceFunctionType)(uintptr_t*, int*, int, uintptr_t, uintptr_t);

        template<typename ContextType, typename RegisterType>
        void FillCpuContextImplCommon(ContextType* out, FillBacktraceFunctionType fillBacktraceFunction, uintptr_t lr, uintptr_t fp) NN_NOEXCEPT
        {
            fillBacktraceFunction(reinterpret_cast<uintptr_t*>(out->backtrace), &out->backtraceCount,
                ContextType::MaxBacktraceCount,
                reinterpret_cast<uintptr_t>(fp),
                reinterpret_cast<uintptr_t>(lr));
            out->SetProgramMappedAddr(reinterpret_cast<RegisterType>(_start));
        }

#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
        void FillCpuContextImpl(fatalsrv::CpuContext* out) NN_NOEXCEPT
        {
            auto lr = reinterpret_cast<uintptr_t>(__builtin_return_address(0));
            auto fp = reinterpret_cast<uintptr_t>(__builtin_frame_address(0));
            out->archType = fatalsrv::CpuContext::ArchType::Aarch64;
            FillCpuContextImplCommon<fatalsrv::Aarch64Context, nn::Bit64>(&out->aarch64, FillCallstack<uint64_t, 0xf>, lr, fp);
        }

        void FillCpuContextFromUserExceptionInfoImpl(fatalsrv::CpuContext* out, const nn::os::UserExceptionInfo* exceptionInfo) NN_NOEXCEPT
        {
            using fatalsrv::Aarch64Context;
            out->archType = fatalsrv::CpuContext::ArchType::Aarch64;
            auto& context = out->aarch64;
            for (int i = 0; i < 31; ++i)
            {
                context.SetValue(static_cast<Aarch64Context::ValueNames>(i), exceptionInfo->detail.r[i]);
            }
            context.SetValue(Aarch64Context::ValueNames::Sp, exceptionInfo->detail.sp);
            context.SetValue(Aarch64Context::ValueNames::Pc, exceptionInfo->detail.pc);
            context.SetValue(Aarch64Context::ValueNames::Pstate, exceptionInfo->detail.pstate);
            context.SetValue(Aarch64Context::ValueNames::Afsr0, exceptionInfo->detail.afsr0);
            context.SetValue(Aarch64Context::ValueNames::Afsr1, exceptionInfo->detail.afsr1);
            context.SetValue(Aarch64Context::ValueNames::Esr, exceptionInfo->detail.esr);
            context.SetValue(Aarch64Context::ValueNames::Far, exceptionInfo->detail.far);
            FillCpuContextImplCommon<fatalsrv::Aarch64Context, nn::Bit64>(&out->aarch64, FillCallstack<uint64_t, 0xf>, exceptionInfo->detail.lr, exceptionInfo->detail.r[29]);
        }

#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
        void FillCpuContextImpl(fatalsrv::CpuContext* out) NN_NOEXCEPT
        {
            auto lr = reinterpret_cast<uintptr_t>(__builtin_return_address(0));
            auto fp = reinterpret_cast<uintptr_t>(__builtin_frame_address(0));
            out->archType = fatalsrv::CpuContext::ArchType::Aarch32;
            FillCpuContextImplCommon<fatalsrv::Aarch32Context, nn::Bit32>(&out->aarch32, FillCallstack<uint32_t, 0x3>, lr, fp);
        }


        void FillCpuContextFromUserExceptionInfoImpl(fatalsrv::CpuContext* out, const nn::os::UserExceptionInfo* exceptionInfo) NN_NOEXCEPT
        {
            using fatalsrv::Aarch32Context;
            out->archType = fatalsrv::CpuContext::ArchType::Aarch32;
            auto& context = out->aarch32;
            for (int i = 0; i < 16; ++i)
            {
                context.SetValue(static_cast<fatalsrv::Aarch32Context::ValueNames>(i) , exceptionInfo->detail.r[i]);
            }
            context.SetValue(Aarch32Context::ValueNames::Pstate, exceptionInfo->detail.pstate);
            context.SetValue(Aarch32Context::ValueNames::Afsr0, exceptionInfo->detail.afsr0);
            context.SetValue(Aarch32Context::ValueNames::Afsr1, exceptionInfo->detail.afsr1);
            context.SetValue(Aarch32Context::ValueNames::Esr, exceptionInfo->detail.esr);
            context.SetValue(Aarch32Context::ValueNames::Far, exceptionInfo->detail.far);
            FillCpuContextImplCommon<fatalsrv::Aarch32Context, nn::Bit32>(&out->aarch32, FillCallstack<uint32_t, 0x3>, exceptionInfo->detail.lr, exceptionInfo->detail.fp);
        }
#else
        void FillCpuContextImpl(fatalsrv::CpuContext*) NN_NOEXCEPT
        {
        }

        void FillCpuContextFromUserExceptionInfoImpl(fatalsrv::CpuContext*, const nn::os::UserExceptionInfo*) NN_NOEXCEPT
        {
        }
#endif

    } // namespace



    void FillCpuContext(fatalsrv::CpuContext* out) NN_NOEXCEPT
    {
        out->Clear();
        FillCpuContextImpl(out);
    }

    void FillCpuContextFromUserExceptionInfo(fatalsrv::CpuContext* out, const nn::os::UserExceptionInfo* exceptionInfo) NN_NOEXCEPT
    {
        out->Clear();
        out->exceptionType = exceptionInfo->exceptionType;
        FillCpuContextFromUserExceptionInfoImpl(out, exceptionInfo);
    }
}}} // namespace nn::fatal
