﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/bgtc.h>

#include "eupldsrv_Worker.h"
#include "eupldsrv_Request.h"
#include "eupldsrv_Transport.h"

namespace nn    {
namespace eupld {
namespace srv   {

namespace {

bool IsTransportablePowerState(nn::psc::PmState state)
NN_NOEXCEPT
{
    return (state == nn::psc::PmState_FullAwake || state == nn::psc::PmState_MinimumAwake);
}

}

nn::TimeSpan       Worker::m_AutoUploadEndTime(0);
bool               Worker::m_AutoUpload = false;
nn::psc::PmState   Worker::m_PmState    = nn::psc::PmState_FullAwake;
nn::os::TimerEvent Worker::m_TimerEvent(nn::os::EventClearMode_AutoClear);
bool               Worker::m_RetryUploadOnAwake = false;

void Worker::HandleNewReport()
NN_NOEXCEPT
{
    nn::erpt::ReportList reportList;
    Request*             pRequest;

    if (!m_AutoUpload || m_AutoUploadEndTime < nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()) || !IsTransportablePowerState(m_PmState))
    {
        return;
    }

    if (m_ReportManager.GetReportList(reportList).IsSuccess())
    {
        if ((pRequest = new Request(nullptr)) == nullptr)
        {
            return;
        }
        if ((pRequest->Initialize(reportList, true)).IsFailure())
        {
            delete pRequest;
        }
    }
}

nn::Result Worker::HandleNewRequest()
NN_NOEXCEPT
{
    Request*   pRequest;
    nn::Result result = ResultSuccess();

    auto bgtcResult = bgtc::NotifyTaskStarting();
    if( bgtcResult.IsFailure() )
    {
        NN_SDK_LOG("[eupld]: bgtc::NotifyTaskStarting() failed %03d-%04d (0x%08x)\n",
            bgtcResult.GetModule(), bgtcResult.GetDescription(), bgtcResult.GetInnerValueForDebug());
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( bgtcResult.IsSuccess() )
        {
            bgtc::NotifyTaskFinished();
        }
    };

    while ((pRequest = Request::GetRequest()) != nullptr)
    {
        pRequest->CompleteRequest(
            IsTransportablePowerState(m_PmState) ?
            (result = Transport::Upload(pRequest)) :
            (result = nn::os::ResultBadRequest())
        );

        if (pRequest->RemoveReference())
        {
            delete pRequest;
        }
    }

    return result;
}

void Worker::UpdateBackOffTimer(nn::Result result)
NN_NOEXCEPT
{
    if( os::ResultBadRequest::Includes(result) )
    {
        NN_SDK_LOG("[eupld]: report upload failed %d-%04d (os::ResultBadRequest)\n", result.GetModule(), result.GetDescription());
        NN_SDK_LOG("[eupld]: will retry when the sytem awakes\n");
        m_TimerEvent.Clear();
        m_RetryUploadOnAwake = true;
    }
    else if (result.IsFailure())
    {
        std::mt19937 engine(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));
        std::uniform_int_distribution<int32_t> dis(1, m_RetryTimeMs / 5);

        // current retry time + 20% randomization
        uint32_t retryTime = m_RetryTimeMs + dis(engine);

        NN_SDK_LOG("[eupld]: report upload failed %d-%04d\n", result.GetModule(), result.GetDescription());
        NN_SDK_LOG("[eupld]: will retry in %d seconds\n", retryTime / 1000);

        m_TimerEvent.StartOneShot(nn::TimeSpan::FromMilliSeconds(retryTime));

        if (m_RetryTimeMs <= MaxRetryTimeMs / 4)
        {
            m_RetryTimeMs *= 4;
        }
    }
    else
    {
        m_RetryTimeMs = InitialRetryTimeMs;
    }
}

void Worker::Loop()
NN_NOEXCEPT
{
    nn::Result result;

    auto& bgtcEvent = bgtc::GetTriggerEvent();

    if ((result = m_ReportManager.Initialize()).IsFailure()  ||
        (result = Transport::SetEnvironmentId()).IsFailure())
    {
        NN_SDK_ASSERT(result.IsSuccess());
        return;
    }

    while (NN_STATIC_CONDITION(true))
    {
        int eventId = nn::os::WaitAny(
                m_PmEvent.GetBase(),
                m_RequestEvent.GetBase(),
                m_ReportManager.GetEventPointer()->GetBase(),
                m_TimerEvent.GetBase(),
                bgtcEvent.GetBase());

        if (!m_Running)
        {
            break;
        }

        switch (eventId)
        {
        case 0:
            m_PmEvent.Clear();
            m_Module.Acknowledge(m_PmState, ResultSuccess());
            Transport::TransportControl(IsTransportablePowerState(m_PmState));
            break;

        case 1:
            m_RequestEvent.Clear();
            m_TimerEvent.Stop();
            m_TimerEvent.Clear();
            UpdateBackOffTimer(
                HandleNewRequest()
            );
            break;

        case 2:
            m_ReportManager.GetEventPointer()->Clear();
            HandleNewReport();
            break;

        case 3:
            m_TimerEvent.Clear();
            HandleNewReport();
            break;

        case 4:
            bgtcEvent.Clear();
            if( m_RetryUploadOnAwake )
            {
                NN_SDK_LOG("[eupld]: retry upload as the sytem awakes\n");
                m_RetryUploadOnAwake = false;
                HandleNewReport();
            }
            break;

        default:
            break;
        }
    }

    m_ReportManager.Finalize();
}

void Worker::HandlePmEvents()
NN_NOEXCEPT
{
    nn::Result         result;
    nn::psc::PmFlagSet flags;

    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Nifm, nn::psc::PmModuleId_Erpt};

    result = m_Module.Initialize(
                nn::psc::PmModuleId_Eupld,
                dependencies,
                sizeof(dependencies) / sizeof(dependencies[0]),
                nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while (m_Running)
    {
        m_Module.GetEventPointer()->Wait();
        m_Module.GetEventPointer()->Clear();
        result = m_Module.GetRequest(&m_PmState, &flags);
        NN_SDK_ASSERT(result.IsSuccess());
        Transport::TransportControl(false);
        m_PmEvent.Signal();
    }
}

nn::Result Worker::SetAutoUpload(bool enable, uint32_t flagValidTimeSeconds)
NN_NOEXCEPT
{
    m_AutoUpload        = enable;
    m_AutoUploadEndTime = nn::TimeSpan::FromSeconds(
        flagValidTimeSeconds +
        nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetSeconds()
    );

    if (m_AutoUpload)
    {
        m_TimerEvent.StartOneShot(nn::TimeSpan::FromSeconds(1));
    }

    return ResultSuccess();
}

nn::Result Worker::GetAutoUpload(bool* pOutIsEnabled, uint32_t* pOutFlagValidTimeSeconds)
NN_NOEXCEPT
{
    *pOutIsEnabled = m_AutoUpload;
    *pOutFlagValidTimeSeconds = static_cast<uint32_t>(m_AutoUploadEndTime.GetSeconds());

    return ResultSuccess();
}

void Worker::ThreadEntry(void* arg)
NN_NOEXCEPT
{
    Worker* p = reinterpret_cast<Worker*>(arg);
    p->Loop();
}

nn::Result Worker::Start()
NN_NOEXCEPT
{
    nn::Result result;

    NN_ABORT_UNLESS_RESULT_SUCCESS(bgtc::Initialize());

    result = nn::os::CreateThread(
                &m_Thread,
                ThreadEntry,
                this,
                m_Stack,
                sizeof(m_Stack),
                NN_SYSTEM_THREAD_PRIORITY(eupld, Worker));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_Running = true;
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(eupld, Worker));
    nn::os::StartThread(&m_Thread);

    return ResultSuccess();
}

void Worker::Stop() NN_NOEXCEPT
{
    if (m_Running)
    {
        m_Running = false;
        m_RequestEvent.Signal();
        nn::os::WaitThread(&m_Thread);
    }
    bgtc::Finalize();
}

void Worker::SignalNewRequest()
NN_NOEXCEPT
{
    m_RequestEvent.Signal();
}

Worker::Worker()
NN_NOEXCEPT:
    m_RequestEvent(nn::os::EventClearMode_AutoClear, false),
    m_PmEvent(nn::os::EventClearMode_AutoClear, false),
    m_RetryTimeMs(InitialRetryTimeMs),
    m_Running(false)
{

}

Worker::~Worker()
NN_NOEXCEPT
{
    Stop();
}

}}}
