﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/eupld/eupld_Request.h>
#include <nn/eupld/sfdl/eupld.sfdl.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

#include "eupld_Session.h"

namespace nn    {
namespace eupld {

nn::Result Request::UploadAll()
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), eupld::ResultNetworkNotAvailable());

    return m_Impl != nullptr ?
           m_Impl->UploadAll() :
           nn::eupld::ResultNotInitialized();
}

nn::Result Request::UploadSelected(const nn::erpt::ReportId* pReportIds, uint32_t reportCount)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), eupld::ResultNetworkNotAvailable());

    nn::sf::InBuffer reportIdArray(reinterpret_cast<const char*>(pReportIds), reportCount * sizeof(nn::erpt::ReportId));

    return m_Impl != nullptr ?
           m_Impl->UploadSelected(reportIdArray) :
           nn::eupld::ResultNotInitialized();
}

nn::Result Request::GetUploadStatus(ReportUploadList& uploadStatus)
NN_NOEXCEPT
{
    nn::sf::OutBuffer status(reinterpret_cast<char*>(&uploadStatus), sizeof(uploadStatus));

    return m_Impl != nullptr ?
           m_Impl->GetUploadStatus(status) :
           nn::eupld::ResultNotInitialized();
}

nn::Result Request::CancelUpload()
NN_NOEXCEPT
{
    return m_Impl != nullptr ?
           m_Impl->CancelUpload() :
           nn::eupld::ResultNotInitialized();
}

nn::Result Request::GetResult()
NN_NOEXCEPT
{
    return m_Impl != nullptr ?
           m_Impl->GetResult() :
           nn::eupld::ResultNotInitialized();
}

nn::os::SystemEvent* Request::GetEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

nn::Result Request::Initialize()
NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;
    nn::Result result;

    if (m_Impl != nullptr)
    {
        return nn::eupld::ResultAlreadyInitialized();
    }

    if ((result = GetRequestObject(&m_Impl)).IsSuccess())
    {
        if ((result = m_Impl->Initialize(&sfHandle)).IsFailure())
        {
            m_Impl = nullptr;
            return result;
        }

        m_SystemEvent.AttachReadableHandle(
                sfHandle.GetOsHandle(),
                sfHandle.IsManaged(),
                nn::os::EventClearMode_AutoClear);

        sfHandle.Detach();
    }

    return result;
}

void Request::Finalize()
NN_NOEXCEPT
{
    if (m_Impl != nullptr)
    {
        m_Impl = nullptr;
        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    }
}

Request::Request()
NN_NOEXCEPT :
    m_Impl(nullptr)
{

}

Request::~Request()
NN_NOEXCEPT
{
    Finalize();
}

}}
