﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/es/es_IETicketService.sfdl.h>
#include <nn/es/es_RightsTypes.h>
#include <nn/es/es_TypesForInner.h>
#include <nn/sf/sf_Types.h>
#include "es_ELicenseList.h"
#include "es_TicketDatabase.h"
#include "es_UserList.h"
#include "es_UsingRightsArray.h"
#include "es_UsingTitleKeyList.h"

namespace nn { namespace es {

class ETicketServiceImpl;
class ActiveRightsContext;

class RightsAvailabilityMaster
{
public:
    void Setup(TicketDatabase* pCommonDatabase, TicketDatabase* pPersonalizedDatabase, TicketMetaRecordDatabase* pTicketMetaRecordDatabase, ELicenseList* pELicenseList) NN_NOEXCEPT
    {
        this->m_pCommonDatabase = pCommonDatabase;
        this->m_pPersonalizedDatabase = pPersonalizedDatabase;
        this->m_pELicenseList = pELicenseList;
        this->m_pTicketMetaRecordDatabase = pTicketMetaRecordDatabase;
    };

    // eTicket および権利チェック関連 ----------
public:
    nn::Result CheckRightsStatusIncludingKeyId(nn::es::RightsStatus outStatusList[], size_t statusCount, const nn::es::RightsIdIncludingKeyId rightsIdList[], size_t rightsIdCount, const nn::es::ELicenseUserId userIdList[], size_t userIdCount, bool onlyAllAccountRights) NN_NOEXCEPT;
    nn::Result CheckRightsStatus(nn::es::RightsStatus outStatusList[], size_t statusCount, const nn::es::RightsId rightsIdList[], size_t rightsIdCount, const nn::es::ELicenseUserId userIdList[], size_t userIdCount, bool onlyAllAccountRights) NN_NOEXCEPT;

private:
    TicketDatabase* m_pCommonDatabase;
    TicketDatabase* m_pPersonalizedDatabase;
    TicketMetaRecordDatabase* m_pTicketMetaRecordDatabase;
    ELicenseList* m_pELicenseList;


    // 権利コンテキスト管理 ----------
public:
    sf::SharedPointer<es::IActiveRightsContext> CreateActiveRightsContext(es::ETicketServiceImpl* p) NN_NOEXCEPT;
    Result UpdateAllRightsIdListsValidity() NN_NOEXCEPT;

private:
    //  es 内部から各権利コンテキストを参照するときは、
    //  m_pContextList[].lock() でポインタを受け取ってアクセスする。
    os::SdkMutex m_MutexContextList;
    std::weak_ptr<ActiveRightsContext> m_pContextList[ActiveRightsContextCountMax];


    // 外部鍵リスト管理 ----------
public:
    Result RegisterTitleKeys(
            const util::Span<const RightsIdIncludingKeyId> rightsIdList,
            const util::Span<const int32_t> keyGenerationList,
            const util::Span<const ELicenseUserId> userIdList
    );
    Result SetAlreadyRegisteredTitleKeys(bool forcibly) NN_NOEXCEPT;
    Result UnregisterAllTitleKeys() NN_NOEXCEPT;

private:
    nn::Result GetTitleKey(nn::es::AesKey* outKey, const nn::es::RightsIdIncludingKeyId& rightsId, int keyGeneration) NN_NOEXCEPT;

private:
    os::SdkMutex m_MutexTitleKeyList;
    UsingTitleKeyList m_TitleKeyList;
};

RightsAvailabilityMaster& GetRightsAvailabilityMaster() NN_NOEXCEPT;

}} // namespace nn::es
