﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <random>
#include <nn/os.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_RsaOaepSha256Encryptor.h>
#include <nn/es/es_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include "es_EncryptionUtil.h"

namespace nn { namespace es {

    Result GenerateRandomBytes(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_RESULT_DO(spl::GenerateRandomBytes(pOutBuffer, bufferSize));
#elif defined(NN_BUILD_CONFIG_OS_WIN)
        std::mt19937 random;
        random.seed(static_cast<unsigned long>(nn::os::GetSystemTick().GetInt64Value()));
        std::uniform_int_distribution<> rand(0, 0xFF);

        for (size_t i = 0; i < bufferSize; i++) {
            static_cast<uint8_t*>(pOutBuffer)[i] = static_cast<uint8_t>(rand(random));
        }
#else
        NN_ABORT("Unexpected platform.");
#endif
        NN_RESULT_SUCCESS;
    }

    size_t CalculateEncryptionTextSizeAes128CbcWithPkcs7Padding(size_t plainTextSize) NN_NOEXCEPT
    {
        return (static_cast<size_t>(floor(plainTextSize / crypto::Aes128CbcEncryptor::BlockSize)) + 1) * crypto::Aes128CbcEncryptor::BlockSize;
    }

    size_t EncryptAes128CbcWithPkcs7Padding(void* pDst, size_t dstSize,
                        const void* pKey, size_t keySize,
                        const void* pIv, size_t ivSize,
                        const void* pSrc, size_t srcSize) NN_NOEXCEPT
    {
        size_t cipherTextSize = CalculateEncryptionTextSizeAes128CbcWithPkcs7Padding(srcSize);
        NN_SDK_REQUIRES_GREATER_EQUAL(dstSize, cipherTextSize);

        memcpy(pDst, pSrc, srcSize);

        // PKCS#7 パディング
        for (size_t i = srcSize; i < cipherTextSize; i++)
        {
            static_cast<uint8_t*>(pDst)[i] = static_cast<uint8_t>(cipherTextSize - srcSize /*- 1*/);
        }

        // AES-CBC 暗号化
        return crypto::EncryptAes128Cbc(
            pDst, cipherTextSize,
            pKey, keySize,
            pIv, ivSize,
            pDst, cipherTextSize
        );
    }

    bool EncryptRsa2048OaepSha256(void* pOutCipher, size_t cipherSize,
        const void* pModulus, size_t modulusSize,
        const void* pPublicExponent, size_t publicExponentSize,
        const void* pRawMessage, size_t rawMessageSize,
        const void* pLabel, size_t labelSize) NN_NOEXCEPT
    {
        uint8_t seed[crypto::Sha256Generator::HashSize];
        NN_RESULT_TRY(GenerateRandomBytes(seed, sizeof(seed)))
            NN_RESULT_CATCH_ALL
            {
                return false;
            }
        NN_RESULT_END_TRY

        return crypto::EncryptRsa2048OaepSha256(
            pOutCipher, cipherSize,
            pModulus, modulusSize,
            pPublicExponent, publicExponentSize,
            pRawMessage, rawMessageSize,
            seed, sizeof(seed),
            pLabel, labelSize
        );
    }
}} // namespace nn::es
