﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/es/es_ELicenseTypesForInner.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace es {

class ELicenseList
{
public:
    bool Insert(ELicenseInfoForSystem license) NN_NOEXCEPT;
    void Commit(const ELicenseArchiveId& archiveId) NN_NOEXCEPT;
    void Revert() NN_NOEXCEPT;
    int Count() NN_NOEXCEPT;

    void Delete(ELicenseOwnerId ownerId) NN_NOEXCEPT;
    void DeleteAll() NN_NOEXCEPT;
    int List(ELicenseId outIdList[], int idCount) NN_NOEXCEPT;
    int Find(ELicenseInfoForSystem outInfoList[], int infoCount, const ELicenseId idList[], int idCount) NN_NOEXCEPT;
    int Find(ELicenseInfoForSystem outInfoList[], int infoCount, RightsId rightsId) NN_NOEXCEPT;

    void EnableELicense() NN_NOEXCEPT
    {
        m_IsELicenseEnabled = true;
    }

    void DisableELicense() NN_NOEXCEPT
    {
        m_IsELicenseEnabled = false;
    }

    /**
    *   @brief  eLicense 配下であるかを取得します。
    */
    bool IsELicenseEnabled() NN_NOEXCEPT
    {
        return m_IsELicenseEnabled;
    }

private:
    // オンメモリで持っている eLicense の数
    static const int CacheRecordCount = 1024;

    //TODO: キャッシュに収まりきらない eLicense をページアウトして、全 eLicense を Traverse するようにする
    template <class FuncT>
    Result Traverse(FuncT func) NN_NOEXCEPT
    {
        bool isEnd = false;
        for (int i = 0; i < CacheRecordCount; i++)
        {
            auto& elicense = m_Cache[i];

            NN_RESULT_DO(func(&isEnd, elicense));
            if (isEnd)
            {
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_SUCCESS;
    }

    struct ELicenseRecord
    {
        enum class State : uint8_t
        {
            Empty,
            Inserted,
            Valid,
        }state;

        ELicenseInfoForSystem license;
    };

    bool m_IsELicenseEnabled = false;
    ELicenseRecord m_Cache[CacheRecordCount];
};

}} // namespace nn::es
