﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nn/err/err_ServiceStatusDownloaderApi.h"

#ifdef _DEBUG

#define ERR_VIEWER_PRINTF(...) NN_SDK_LOG(__VA_ARGS__)


#else

#define ERR_VIEWER_PRINTF(...) (void)0

#endif

#define CURL_EASY_DO(exp) \
    do                                                                                   \
    {                                                                                    \
        m_ErrorCode = (exp);                                                             \
                                                                                         \
        if (m_ErrorCode != CURLE_OK)                                                     \
        {                                                                                \
           SetLastResult(detail::HandleCurlError(m_ErrorCode));                          \
        }                                                                                \
    }                                                                                    \
    while (NN_STATIC_CONDITION(0))

namespace {
#if defined( NN_BUILD_CONFIG_OS_WIN )
    //Windows環境ではSDKチームの動作確認用に使用するので td1 固定
    const char* ServiceStatusJsonUrl =  "https://Service-status-td1.cdn.nintendo.net/service_status_hac.json";
#else
    const char* ServiceStatusJsonUrl =  "https://Service-status-%.cdn.nintendo.net/service_status_hac.json";
#endif
    bool g_IsKeepAlive = true;
    const int UrlSize = 128;
    const int DownloadLowSpeedTimeOutSec = 120;
    const char* ErrViewerCurlPrivateJason = "s";
    const int MaxDataSize = 921600;
}

namespace nn { namespace err {
namespace detail
{
    //CURLのエラーコードを nn::Resultへ変換します。
    nn::Result HandleCurlError(CURLcode code) NN_NOEXCEPT;

    // Curl の書き込み関数です。
    size_t WriteFunction(char* pBuffer, size_t size, size_t nmemb, void* pUserData) NN_NOEXCEPT;

    // CURLOPT_SSL_CTX_FUNCTIONに対するCallback関数。
    CURLcode SslCtxFunction(CURL* pCurl, void* pSslContext, void* pParam) NN_NOEXCEPT;

    //Curl の進捗関数です。
    int ProgressFunction(void* pUserData, double totalDownloaded,
        double nowDownloaded, double totalUploaded, double nowUploaded) NN_NOEXCEPT;
}

    void ServiceStatusDownloader::Initialize() NN_NOEXCEPT
    {
        m_IsCanceled = false;
        nn::Result result = m_SslContext.Create(nn::ssl::Context::SslVersion_Auto);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return;
    }

    void ServiceStatusDownloader::Finalize() NN_NOEXCEPT
    {
        if (ServiceStatusDownloader::m_LastResult.IsSuccess())
        {
            m_SslContext.Destroy();
        }
    }

    nn::Result ServiceStatusDownloader::Download(char* pBuffer, size_t size) NN_NOEXCEPT
    {
        DownloadContext context;
        m_pDownloadContext = &context;
        ResetLastResult();
        m_IsCanceled = false;
        m_ErrorCode = CURLE_OK;

        CreateDownloadContext(m_pDownloadContext, pBuffer, size);

        // DownloadFile を呼び出す時のみ進捗コールバックを設定する
        CURL_EASY_DO(curl_easy_setopt(m_pDownloadContext->pCurl, CURLOPT_NOPROGRESS, 0));
        CURL_EASY_DO(curl_easy_setopt(m_pDownloadContext->pCurl, CURLOPT_PROGRESSDATA, &m_IsCanceled));
        CURL_EASY_DO(curl_easy_setopt(m_pDownloadContext->pCurl, CURLOPT_PROGRESSFUNCTION, detail::ProgressFunction));
        CURL_EASY_DO(curl_easy_perform(m_pDownloadContext->pCurl));

        if (m_ErrorCode != CURLE_OK)
        {
            ERR_VIEWER_PRINTF("[ERR_VIWER] DownloadFile() failed. (errorCode = %d)\n", m_ErrorCode);
        }

        DestroyDownloadContext(m_pDownloadContext);
        m_pDownloadContext = nullptr;

        return GetLastResult();
    }

    void ServiceStatusDownloader::Cancel() NN_NOEXCEPT
    {
        m_IsCanceled = true;
    }

    nn::Result ServiceStatusDownloader::CreateDownloadContext(DownloadContext* pContext, void* pBuffer, size_t size) NN_NOEXCEPT
    {
        ResetLastResult();

        char errViewerUrl[UrlSize] = {};
        util::SNPrintf(errViewerUrl, UrlSize, ServiceStatusJsonUrl);

        if (pContext == NULL)
        {
            ERR_VIEWER_PRINTF("[ERR_VIEWER] CreateDownloadContext() failed. (pContext == NULL)\n");
            return GetLastResult();
        }

        pContext->pCurl = curl_easy_init();
        if (pContext->pCurl == NULL)
        {
            ERR_VIEWER_PRINTF("[ERR_VIEWER] CreateDownloadContext() failed. (pCurl == NULL)\n");
            SetLastResult(detail::HandleCurlError(m_ErrorCode));
            return GetLastResult();
        }

        pContext->pBuffer = pBuffer;
        pContext->readSize = 0;

        ERR_VIEWER_PRINTF("[ERR_VIEWER] URL = %s\n", errViewerUrl);

        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_URL, errViewerUrl));
        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_PRIVATE, ErrViewerCurlPrivateJason));
        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_MAXFILESIZE, size));


        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_WRITEDATA, pContext));
        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_WRITEFUNCTION, detail::WriteFunction));

        // 通信の無い期間が IDBE_DOWNLOAD_LOW_SPEED_TIME_OUT_SEC 秒続いたらタイムアウトとする。
        // ただし CURLOPT_LOW_SPEED_LIMIT は 0 には設定できない(無効となってしまう)ため、最小値の 1 を設定する。
        // これにより「通信速度が 1B/s の期間が IDBE_DOWNLOAD_LOW_SPEED_TIME_OUT_SEC 秒続いたらタイムアウト」となる。
        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_LOW_SPEED_LIMIT, 1));
        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_LOW_SPEED_TIME, DownloadLowSpeedTimeOutSec));

        if (g_IsKeepAlive)
        {
            pContext->pHeader = curl_slist_append(NULL, "Connection: Keep-Alive");

            CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_HTTPHEADER, pContext->pHeader));
        }
        else
        {
            pContext->pHeader = NULL;
        }

        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_VERIFYDATE, 0));

        if (m_ErrorCode != CURLE_OK)
        {
            ERR_VIEWER_PRINTF("[ERR_VIEWER] curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_VERIFYDATE, 0) failed. (errorCode = %d)\n", m_ErrorCode);
        }

        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_CTX_FUNCTION, detail::SslCtxFunction));

        if (m_ErrorCode != CURLE_OK)
        {
            ERR_VIEWER_PRINTF("[ERR_VIEWER] curl_easy_setopt(CURLOPT_SSL_CTX_DATA) failed. (errorCode = %d)\n", m_ErrorCode);
            DestroyDownloadContext(pContext);
        }

        return GetLastResult();
    }

    void ServiceStatusDownloader::DestroyDownloadContext(DownloadContext* pContext) NN_NOEXCEPT
    {
        if (pContext == NULL)
        {
            return;
        }

        if (pContext->pHeader != NULL)
        {
            curl_slist_free_all(pContext->pHeader);
            pContext->pHeader = NULL;
        }

        if (pContext->pCurl != NULL)
        {
            curl_easy_cleanup(pContext->pCurl);
            pContext->pCurl = NULL;
        }
    }

    nn::Result ServiceStatusDownloader::CheckDownloadResult() const NN_NOEXCEPT
    {
        return GetLastResult();
    }

    void ServiceStatusDownloader::SetLastResult(nn::Result result) NN_NOEXCEPT
    {
        m_LastResult = result;
    }

    nn::Result ServiceStatusDownloader::GetLastResult() const NN_NOEXCEPT
    {
        return m_LastResult;
    }

    void ServiceStatusDownloader::ResetLastResult() NN_NOEXCEPT
    {
        m_LastResult = nn::ResultSuccess();
    }

    CURLcode detail::SslCtxFunction(CURL* pCurl, void* pSslContext, void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pSslContext);
        NN_UNUSED(pParam);
        nn::ssl::CertStoreId deviceCertId;
        nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

        if (pContext->Create(nn::ssl::Context::SslVersion_Auto).IsSuccess())
        {
            if (pContext->RegisterInternalPki(&deviceCertId, nn::ssl::Context::InternalPki_DeviceClientCertDefault).IsSuccess())
            {
                return CURLE_OK;
            }
        }
        return CURLE_ABORTED_BY_CALLBACK;
    }

    size_t detail::WriteFunction(char* pBuffer, size_t size, size_t nmemb, void* pUserData) NN_NOEXCEPT
    {
        size_t readSize = size * nmemb;

        DownloadContext* pContext = static_cast<DownloadContext*>(pUserData);

        size_t maxSize = MaxDataSize;

        if (pContext->readSize + readSize > maxSize)
        {
            return 0;
        }

        memcpy(static_cast<uint8_t*>(pContext->pBuffer) + pContext->readSize, pBuffer, readSize);
        pContext->readSize += readSize;

        return readSize;
    }

    int detail::ProgressFunction(void* pUserData, double totalDownloaded, double nowDownloaded,
                                 double totalUploaded, double nowUploaded) NN_NOEXCEPT
    {
        NN_UNUSED(totalDownloaded);
        NN_UNUSED(nowDownloaded);
        NN_UNUSED(totalUploaded);
        NN_UNUSED(nowUploaded);

        bool* pIsCanceled = static_cast<bool*>(pUserData);

        if (*pIsCanceled)
        {
            return 1;
        }

        return 0;
    }
}}
