﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_ApiForErrorViewer.h>
#include <nn/err/err_ErrorInfo.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_SystemApi.h>
#include <nn/err/detail/err_SystemData.h>
#include <nn/err/detail/err_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/system/settings_Language.h>
#include <algorithm>

namespace nn { namespace err {

    namespace
    {
        void InvalidateUiSettings(ErrorInfo::UiSettings* outValue) NN_NOEXCEPT
        {
            outValue->buttonCount = 0;
            outValue->buttonAction[0] = 0;
            outValue->message[0] = 0;
            outValue->messageLength = 0;
        }

        void SetCloseButton(ErrorInfo::UiSettings* outValue, int buttonIndex, const settings::LanguageCode& languageCode) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(outValue);
            NN_SDK_ASSERT_RANGE(buttonIndex, 0, 3);
            outValue->buttonAction[buttonIndex] = 0; // = とじる
            detail::ReadMessageFile(outValue->buttonMessage[buttonIndex], ErrorInfo::UiSettings::ButtonMessageLengthMax, "Close", languageCode);
        }

        void SetShowFullScreenViewButton(ErrorInfo::UiSettings* outValue, int buttonIndex, const settings::LanguageCode& languageCode) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(outValue);
            NN_SDK_ASSERT_RANGE(buttonIndex, 0, 3);
            outValue->buttonAction[buttonIndex] = 1; // = くわしく
            detail::ReadMessageFile(outValue->buttonMessage[buttonIndex], ErrorInfo::UiSettings::ButtonMessageLengthMax, "Detail", languageCode);
        }

        void SetOkButton(ErrorInfo::UiSettings* outValue, int buttonIndex, const settings::LanguageCode& languageCode) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(outValue);
            NN_SDK_ASSERT_RANGE(buttonIndex, 0, 3);
            outValue->buttonAction[buttonIndex] = 0; // = OK（挙動としては「とじる」と同じ）
            detail::ReadMessageFile(outValue->buttonMessage[buttonIndex], ErrorInfo::UiSettings::ButtonMessageLengthMax, "Ok", languageCode);
        }

        bool HasMessage(const char* message) NN_NOEXCEPT
        {
            return (message != nullptr) && message[0] != '\0';
        }

        void SetButtonMessage(ErrorInfo::UiSettings* outValue, char16_t* inputBuffer, int inputBufferLength) NN_NOEXCEPT
        {
            int buttonIndex = 0;
            int characterIndex = 0;
            for( int i = 0; i < inputBufferLength; i++ )
            {
                if( inputBuffer[i] == 0 )
                {
                    outValue->buttonMessage[buttonIndex][characterIndex] = 0;
                    break;
                }
                if( inputBuffer[i] == 0x0A ) // 改行
                {
                    outValue->buttonMessage[buttonIndex][characterIndex] = 0;
                    buttonIndex++;
                    characterIndex = 0;
                    continue;
                }
                outValue->buttonMessage[buttonIndex][characterIndex] = inputBuffer[i];
                characterIndex++;
            }
        }

        bool IsSystemError(err::ErrorType errorType) NN_NOEXCEPT
        {
            return (errorType == ErrorType::SystemData) || (errorType == ErrorType::SystemError)
                || (errorType == ErrorType::RecordedSystemData) || (errorType == ErrorType::RecordedSystemError);
        }

        bool IsApplicationError(err::ErrorType errorType) NN_NOEXCEPT
        {
            return (errorType == ErrorType::ApplicationError) || (errorType == ErrorType::RecordedApplicationError);
        }
    }

ErrorInfo::ErrorInfo(const void* startupParam) NN_NOEXCEPT
    : m_Result(nn::ResultSuccess())
{
    m_ErrorType = GetErrorType(startupParam);

    switch( m_ErrorType )
    {
    case ErrorType::SystemData:
        {
            auto pParam = reinterpret_cast<const ErrorViewerStartupParamForSystemData*> (startupParam);
            if( !pParam->isErrorCode )
            {
                m_Result = pParam->result;
            }
            ReadSystemData(pParam->GetErrorCode());
        }
        break;

    case ErrorType::SystemError:
        {
            auto param = reinterpret_cast<const ErrorViewerStartupParamForSystemError*>(startupParam);
            m_ErrorCode = param->errorCode;
            InitializeWithMessages(param->dialogMessage, param->fullScreenMessage, param->languageCode);
            m_IsServerError = false;
            m_AttributeFlagSetValue = 0;
        }
        break;

    case ErrorType::ApplicationError:
        {
            NN_SDK_ASSERT(false, "Use `ErrorInfo(const ErrorViewerStartupParamHolder& startupParam, const ns::ApplicationErrorCodeCategory& category)` instead.\n");
        }
        break;

    case ErrorType::Eula:
        {
            NN_SDK_ASSERT(false, "ErrorInfo is not available for ErrorType::Eula.\n");
        }
        break;

    case ErrorType::ParentalControl:
        {
            NN_SDK_ASSERT(false, "ErrorInfo is not available for ErrorType::ParentalControl.\n");
        }
        break;

    case ErrorType::RecordedSystemData:
        {
            ReadSystemData(reinterpret_cast<const ErrorViewerStartupParamForRecordedSystemData*>(startupParam)->errorCode);
        }
        break;

    case ErrorType::RecordedSystemError:
        {
            auto param = reinterpret_cast<const ErrorViewerStartupParamForRecordedSystemError*>(startupParam);
            m_ErrorCode = param->errorCode;
            InitializeWithMessageForRecordedError(param->message);
            m_IsServerError = false;
            m_AttributeFlagSetValue = 0;
        }
        break;

    case ErrorType::RecordedApplicationError:
        {
            auto param = reinterpret_cast<const ErrorViewerStartupParamForRecordedApplicationError*>(startupParam);
            util::Strlcpy(m_ApplicationErrorCodeCategory.value, param->applicationErrorCodeCategory.value, sizeof(m_ApplicationErrorCodeCategory.value));
            m_ApplicationErrorCodeNumber = param->applicationErrorCodeNumber;
            InitializeWithMessageForRecordedError(param->message);
            m_IsServerError = false;
            m_AttributeFlagSetValue = 0;
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

ErrorInfo::ErrorInfo(const void* startupParam, const ns::ApplicationErrorCodeCategory& category) NN_NOEXCEPT
    : m_ErrorType(ErrorType::ApplicationError)
    , m_ApplicationErrorCodeCategory(category)
    , m_IsServerError(false)
    , m_AttributeFlagSetValue(0)
    , m_Result(nn::ResultSuccess())
{
    NN_SDK_REQUIRES(GetErrorType(startupParam) == ErrorType::ApplicationError);

    auto param = reinterpret_cast<const ErrorViewerStartupParamForApplicationError*>(startupParam);
    m_ApplicationErrorCodeNumber = param->applicationErrorCodeNumber;
    InitializeWithMessages(param->dialogMessage, param->fullScreenMessage, param->languageCode);
}

ErrorInfo::ErrorInfo(const nn::err::ErrorCode& errorCode) NN_NOEXCEPT
    : m_ErrorType(ErrorType::SystemData)
    , m_Result(nn::ResultSuccess())
{
    NN_SDK_REQUIRES(errorCode.IsValid());

    ReadSystemData(errorCode);
}

ErrorInfo::ErrorInfo(const nn::Result& result) NN_NOEXCEPT
    : m_ErrorType(ErrorType::SystemData)
    , m_Result(result)
{
    NN_SDK_REQUIRES(result.IsFailure());

    auto errorCode = ErrorCode{};
    errorCode.category = static_cast<ErrorCodeCategory>(result.GetModule() + 2000);
    errorCode.number = static_cast<ErrorCodeNumber>(result.GetDescription());
    ReadSystemData(errorCode);
}

ErrorInfo::ErrorInfo(const ErrorCode& errorCode, const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT
    : m_ErrorType(ErrorType::SystemError)
    , m_ErrorCode(errorCode)
    , m_IsServerError(false)
    , m_AttributeFlagSetValue(0)
    , m_Result(nn::ResultSuccess())
{
    NN_SDK_REQUIRES(errorCode.IsValid());

    m_ErrorCode = errorCode;
    InitializeWithMessages(dialogMessage, fullScreenMessage, languageCode);
}

ErrorInfo::ErrorInfo(const ns::ApplicationErrorCodeCategory& category, ApplicationErrorCodeNumber number,
    const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT
    : m_ErrorType(ErrorType::ApplicationError)
    , m_ApplicationErrorCodeCategory(category)
    , m_ApplicationErrorCodeNumber(number)
    , m_IsServerError(false)
    , m_AttributeFlagSetValue(0)
    , m_Result(nn::ResultSuccess())
{
    InitializeWithMessages(dialogMessage, fullScreenMessage, languageCode);
}

bool ErrorInfo::IsValid() const NN_NOEXCEPT
{
    if( IsSystemError(m_ErrorType) )
    {
        return m_ErrorCode.IsValid();
    }
    if( IsApplicationError(m_ErrorType) )
    {
        return (nn::util::Strnlen(m_ApplicationErrorCodeCategory.value, sizeof(m_ApplicationErrorCodeCategory.value)) >= 1)
            && (m_ApplicationErrorCodeNumber <= 9999);
    }
    return false;
}

void ErrorInfo::GetErrorCodeString(char16_t* codeString, size_t maxLength) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(codeString);
    NN_SDK_REQUIRES_GREATER(maxLength, 0u);
    NN_SDK_REQUIRES(IsValid());

    char charBuffer[ErrorCode::StringLengthMax];

    if( IsSystemError(m_ErrorType) )
    {
        util::TSNPrintf(charBuffer, ErrorCode::StringLengthMax, "%04d-%04d", m_ErrorCode.category, m_ErrorCode.number);
    }
    else if( IsApplicationError(m_ErrorType) )
    {
        util::TSNPrintf(charBuffer, ErrorCode::StringLengthMax, "2-%s-%04d", m_ApplicationErrorCodeCategory.value, m_ApplicationErrorCodeNumber);
    }
    else
    {
        // IsValie() == false になるのでここには到達しない
        NN_SDK_ASSERT(false);
    }

    auto result = util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(codeString), static_cast<int>(maxLength), charBuffer, nn::util::Strnlen(charBuffer, ErrorCode::StringLengthMax) + 1);
    if( result != util::CharacterEncodingResult_Success )
    {
        NN_DETAIL_ERR_WARN("GetErrorCodeString : util::ConvertStringUtf8ToUtf16Native failed. (%d)\n", result);
    }
}

bool ErrorInfo::IsServerError() const NN_NOEXCEPT
{
    return m_IsServerError;
}

const char* ErrorInfo::GetServerCode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsServerError);
    return m_ServerCode;
}

bool ErrorInfo::IsDialogViewSupported() const NN_NOEXCEPT
{
    return nn::util::Strnlen(m_DialogViewUiSettings.message, sizeof(ErrorInfo::UiSettings::MessageLengthMax)) > 0;
}

bool ErrorInfo::IsFullScreenViewSupported() const NN_NOEXCEPT
{
    return nn::util::Strnlen(m_FullScreenViewUiSettings.message, sizeof(ErrorInfo::UiSettings::MessageLengthMax)) > 0;
}

const ErrorInfo::UiSettings& ErrorInfo::GetDialogViewUiSettings() const NN_NOEXCEPT
{
    return m_DialogViewUiSettings;
}

const ErrorInfo::UiSettings& ErrorInfo::GetFullScreenViewUiSettings() const NN_NOEXCEPT
{
    return m_FullScreenViewUiSettings;
}

uint32_t ErrorInfo::GetAttributeFlagSetValue() const NN_NOEXCEPT
{
    return m_AttributeFlagSetValue;
}

nn::Result ErrorInfo::GetResult() const NN_NOEXCEPT
{
    return m_Result;
}

void ErrorInfo::InitializeWithMessages(const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT
{
    InvalidateUiSettings(&m_DialogViewUiSettings);
    InvalidateUiSettings(&m_FullScreenViewUiSettings);

    if( HasMessage(dialogMessage) )
    {
        if( HasMessage(fullScreenMessage) )
        {
            // ダイアログ表示 + 全画面表示
            auto dialogMessageConvertResult = nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(m_DialogViewUiSettings.message), ErrorInfo::UiSettings::MessageLengthMax, dialogMessage);
            NN_SDK_ASSERT_EQUAL(util::CharacterEncodingResult_Success, dialogMessageConvertResult);
            NN_UNUSED(dialogMessageConvertResult);
            m_DialogViewUiSettings.messageLength = util::Strnlen(m_DialogViewUiSettings.message, UiSettings::MessageLengthMax);
            m_DialogViewUiSettings.buttonCount = 2;
            SetShowFullScreenViewButton(&m_DialogViewUiSettings, 0, languageCode);
            SetCloseButton(&m_DialogViewUiSettings, 1, languageCode);

            auto fullMessageConvertResult = nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(m_FullScreenViewUiSettings.message), ErrorInfo::UiSettings::MessageLengthMax, fullScreenMessage);
            NN_SDK_ASSERT_EQUAL(util::CharacterEncodingResult_Success, fullMessageConvertResult);
            NN_UNUSED(fullMessageConvertResult);
            m_FullScreenViewUiSettings.messageLength = util::Strnlen(m_FullScreenViewUiSettings.message, UiSettings::MessageLengthMax);
            m_FullScreenViewUiSettings.buttonCount = 1;
            SetOkButton(&m_FullScreenViewUiSettings, 0, languageCode);
        }
        else
        {
            // ダイアログ表示のみ
            auto dialogMessageConvertResult = nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(m_DialogViewUiSettings.message), ErrorInfo::UiSettings::MessageLengthMax, dialogMessage);
            NN_SDK_ASSERT_EQUAL(util::CharacterEncodingResult_Success, dialogMessageConvertResult);
            NN_UNUSED(dialogMessageConvertResult);
            m_DialogViewUiSettings.messageLength = util::Strnlen(m_DialogViewUiSettings.message, UiSettings::MessageLengthMax);
            m_DialogViewUiSettings.buttonCount = 1;
            SetOkButton(&m_DialogViewUiSettings, 0, languageCode);
        }
    }
    else if( HasMessage(fullScreenMessage) )
    {
        // 全画面表示のみ
        auto fullMessageConvertResult = nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(m_FullScreenViewUiSettings.message), ErrorInfo::UiSettings::MessageLengthMax, fullScreenMessage);
        NN_SDK_ASSERT_EQUAL(util::CharacterEncodingResult_Success, fullMessageConvertResult);
        NN_UNUSED(fullMessageConvertResult);
        m_FullScreenViewUiSettings.messageLength = util::Strnlen(m_FullScreenViewUiSettings.message, UiSettings::MessageLengthMax);
        m_FullScreenViewUiSettings.buttonCount = 1;
        SetOkButton(&m_FullScreenViewUiSettings, 0, languageCode);
    }
    else
    {
        NN_DETAIL_ERR_WARN("Both dialogMessage and fullScreenMessage are empty.\n");
    }
}

void ErrorInfo::InitializeWithMessageForRecordedError(const char* message) NN_NOEXCEPT
{
    InvalidateUiSettings(&m_DialogViewUiSettings);
    InvalidateUiSettings(&m_FullScreenViewUiSettings);

    // エラーを記録する際、メッセージについては全画面表示用のメッセージがあれば全画面表示用を、なければダイアログ表示用のメッセージという形で片方のみを保存している。
    // 表示用に再構築する際は、どちらの場合も全画面表示用のメッセージとして設定する。
    // （ボタン等の情報は使用しないため保存していないのでメッセージのみ）
    auto messageConvertResult = nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(m_FullScreenViewUiSettings.message), ErrorInfo::UiSettings::MessageLengthMax, message);
    NN_SDK_ASSERT_EQUAL(util::CharacterEncodingResult_Success, messageConvertResult);
    NN_UNUSED(messageConvertResult);
    m_FullScreenViewUiSettings.messageLength = util::Strnlen(m_FullScreenViewUiSettings.message, UiSettings::MessageLengthMax);
}

void ErrorInfo::ReadSystemData(const ErrorCode& errorCode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(errorCode.IsValid(), "'errorCode' is invalid");

    m_ErrorCode = errorCode;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const nn::ncm::SystemDataId ErrorMessageSystemDataId = { NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE };
    auto mountResult = nn::fs::MountSystemData(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, ErrorMessageSystemDataId);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage(0x%08x) due to error 0x%08x.\n", NN_ERR_SYSTEM_DATA_ID_OF_ERROR_MESSAGE, mountResult.GetInnerValueForDebug());
        InvalidateUiSettings(&m_DialogViewUiSettings);
        InvalidateUiSettings(&m_FullScreenViewUiSettings);
        return;
    }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    // TORIAEZU: デバッグ用に PC 上のディレクトリをマウント
    const char* testHostDirectory = "C:/Temp/ErrorMessage";
    auto mountResult = nn::fs::MountHost(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME, testHostDirectory);
    if( mountResult.IsFailure() )
    {
        NN_DETAIL_ERR_ERROR("Failed to mount ErrorMessage test directory.\n");
        InvalidateUiSettings(&m_DialogViewUiSettings);
        InvalidateUiSettings(&m_FullScreenViewUiSettings);
        return;
    }
#else
#error "unsupported os"
#endif
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(NN_ERR_DETAIL_SYSTEM_DATA_MOUNT_NAME); };

    // メッセージデータを読み込む先のエラーコード
    ErrorCode messageDataErrorCode = {};

    if( detail::ErrorMessageDataExists(errorCode) )
    {
        // カテゴリ + 番号に一致するメッセージがある → XXXX-YYYY
        messageDataErrorCode = errorCode;
    }
    else if( detail::DefaultErrorMessageDataExists(errorCode.category) )
    {
        // カテゴリ + 番号に一致するメッセージはないが、カテゴリは存在し、そのモジュールがデフォルトメッセージ（デスクリプション番号 9999）を定義している場合 → XXXX-9999
        messageDataErrorCode.category = errorCode.category;
        messageDataErrorCode.number = detail::DefaultMessageErrorNumberId;
    }
    else
    {
        // カテゴリ + 番号に一致するメッセージがなく、カテゴリのデフォルトメッセージも存在しない（カテゴリ自体が存在しない場合も含む） → 2999-9999
        messageDataErrorCode = detail::CategoryDefaultNotFoundErrorCode;
    }

    const size_t FilePathLengthMax = 32;
    char errorMessageDataCommonFile[FilePathLengthMax];
    detail::MakeErrorInfoCommonFilePath(errorMessageDataCommonFile, FilePathLengthMax, messageDataErrorCode);

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, errorMessageDataCommonFile, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    detail::CommonData commonData = {};
    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, fileHandle));
    NN_ABORT_UNLESS_LESS_EQUAL(static_cast<size_t>(fileSize), sizeof(commonData));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, &commonData, static_cast<size_t>(fileSize)));

    m_DialogViewUiSettings.buttonCount = commonData.dialogButtonCount;
    m_DialogViewUiSettings.buttonAction[0] = commonData.dialogButtonAction[0];
    m_DialogViewUiSettings.buttonAction[1] = commonData.dialogButtonAction[1];
    m_DialogViewUiSettings.buttonAction[2] = commonData.dialogButtonAction[2];
    m_FullScreenViewUiSettings.buttonCount = commonData.fullScreenButtonCount;
    m_FullScreenViewUiSettings.buttonAction[0] = commonData.fullScreenButtonAction[0];
    m_FullScreenViewUiSettings.buttonAction[1] = commonData.fullScreenButtonAction[1];
    m_FullScreenViewUiSettings.buttonAction[2] = commonData.fullScreenButtonAction[2];
    m_AttributeFlagSetValue = commonData.attributeFlagSetValue;

    m_IsServerError = (commonData.server == 1);
    nn::util::Strlcpy(m_ServerCode, commonData.serverCode, ErrorInfo::ServerCodeLengthMax);

    nn::settings::LanguageCode langCode;
    nn::settings::GetLanguageCode(&langCode);

    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadMessageFile(m_DialogViewUiSettings.message, &m_DialogViewUiSettings.messageLength, ErrorInfo::UiSettings::MessageLengthMax,
        commonData.dialogMessageReference.IsValid() ? commonData.dialogMessageReference : messageDataErrorCode, langCode, detail::MessageKind::Dialog));
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadMessageFile(m_FullScreenViewUiSettings.message, &m_FullScreenViewUiSettings.messageLength, ErrorInfo::UiSettings::MessageLengthMax,
        commonData.fullScreenMessageReference.IsValid() ? commonData.fullScreenMessageReference : messageDataErrorCode, langCode, detail::MessageKind::FullScreen));

    // ボタン用メッセージは1ファイルに1行1ボタンなので、一時読み込み用バッファーに読み込んでから1行ずつコピーする
    const int ButtonMessageBufferLength = UiSettings::ButtonCountMax * (ErrorInfo::UiSettings::ButtonMessageLengthMax);
    char16_t btnMessageBuffer[ButtonMessageBufferLength];

    int messageLength;
    memset(btnMessageBuffer, 0, ButtonMessageBufferLength * sizeof(char16_t));
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadMessageFile(btnMessageBuffer, &messageLength, ButtonMessageBufferLength,
        commonData.dialogButtonMessageReference.IsValid() ? commonData.dialogButtonMessageReference : messageDataErrorCode, langCode, detail::MessageKind::DialogButton));
    SetButtonMessage(&m_DialogViewUiSettings, btnMessageBuffer, ButtonMessageBufferLength);

    memset(btnMessageBuffer, 0, ButtonMessageBufferLength * sizeof(char16_t));
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadMessageFile(btnMessageBuffer, &messageLength, ButtonMessageBufferLength,
        commonData.fullScreenButtonMessageReference.IsValid() ? commonData.fullScreenButtonMessageReference : messageDataErrorCode, langCode, detail::MessageKind::FullScreenButton));
    SetButtonMessage(&m_FullScreenViewUiSettings, btnMessageBuffer, ButtonMessageBufferLength);
}

}}
