﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_ApplicationErrorArg.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/settings_Language.h>

namespace nn { namespace err {

ApplicationErrorArg::ApplicationErrorArg() NN_NOEXCEPT
{
    auto& data = util::Get(m_Data);
    data.common.errorType = ErrorType::ApplicationError;
    data.common.isJumpEnabled = true;
    data.applicationErrorCodeNumber = 0;
    util::Strlcpy(data.dialogMessage, "", ApplicationErrorArg::MessageLengthMax);
    util::Strlcpy(data.fullScreenMessage, "", ApplicationErrorArg::MessageLengthMax);
}

ApplicationErrorArg::ApplicationErrorArg(ApplicationErrorCodeNumber number, const char* dialogViewMessage, const char* fullScreenViewMessage, const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(number, 9999u);
    NN_SDK_REQUIRES_NOT_NULL(dialogViewMessage);
    NN_SDK_REQUIRES_NOT_NULL(fullScreenViewMessage);

    auto& data = util::Get(m_Data);
    data.common.errorType = ErrorType::ApplicationError;
    data.common.isJumpEnabled = true;
    data.applicationErrorCodeNumber = number;

    util::Strlcpy(data.dialogMessage, dialogViewMessage, ApplicationErrorArg::MessageLengthMax);
    util::Strlcpy(data.fullScreenMessage, fullScreenViewMessage, ApplicationErrorArg::MessageLengthMax);
    util::Strlcpy(data.languageCode.string, languageCode.string, sizeof(data.languageCode.string));
}

void ApplicationErrorArg::SetApplicationErrorCodeNumber(ApplicationErrorCodeNumber number) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(number, 9999u);
    util::Get(m_Data).applicationErrorCodeNumber = number;
}

void ApplicationErrorArg::SetDialogMessage(const char* message) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(message);
    util::Strlcpy(util::Get(m_Data).dialogMessage, message, ApplicationErrorArg::MessageLengthMax);
}

void ApplicationErrorArg::SetFullScreenMessage(const char* message) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(message);
    util::Strlcpy(util::Get(m_Data).fullScreenMessage, message, ApplicationErrorArg::MessageLengthMax);
}

void ApplicationErrorArg::SetLanguageCode(const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
{
    util::Strlcpy(util::Get(m_Data).languageCode.string, languageCode.string, sizeof(util::Get(m_Data).languageCode.string));
}

ApplicationErrorCodeNumber ApplicationErrorArg::GetApplicationErrorCodeNumber() const NN_NOEXCEPT
{
    return util::Get(m_Data).applicationErrorCodeNumber;
}

const char* ApplicationErrorArg::GetDialogMessage() const NN_NOEXCEPT
{
    return util::Get(m_Data).dialogMessage;
}

const char* ApplicationErrorArg::GetFullScreenMessage() const NN_NOEXCEPT
{
    return util::Get(m_Data).fullScreenMessage;
}

nn::settings::LanguageCode ApplicationErrorArg::GetLanguageCode() const NN_NOEXCEPT
{
    return util::Get(m_Data).languageCode;
}

}}
