﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt/erpt_Result.h>

#include "erptsrv_Allocator.h"
#include "erptsrv_Stream.h"

namespace nn   {
namespace erpt {
namespace srv  {

#define Min(a,b) ((a) < (b) ? (a) : (b))

bool Stream::m_FsAvailable = true;

void Stream::EnableFsAccess(bool enable)
NN_NOEXCEPT
{
    m_FsAvailable = enable;
}

nn::Result Stream::ReadStream(uint32_t* pReadLength, uint8_t* pBuffer, uint32_t length)
NN_NOEXCEPT
{
    nn::Result         result         = ResultSuccess();
    nn::fs::FileHandle fileHandle     = {nullptr};
    uint32_t           readCount      = 0;
    bool               readStreamOpen = false;
    size_t             fsReadCount;
    uint32_t           count;

    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    if (!m_Initialized || m_StreamMode != StreamMode_Read)
    {
        return nn::erpt::ResultNotInitialized();
    }

    if (pReadLength == nullptr || pBuffer == nullptr)
    {
        return nn::erpt::ResultInvalidArgument();
    }

    while (length > 0)
    {
        if ((count = Min(m_BufferCount - m_BufferPosition, length)) > 0)
        {
            std::memcpy(pBuffer, m_pBuffer + m_BufferPosition, count);

            pBuffer          += count;
            length           -= count;
            readCount        += count;
            m_BufferPosition += count;
        }
        else
        {
            // Limitation imposed by FS team: all files in save data area must
            // be closed in order for CommitSaveData() to succeeed.
            // Workaround: Open->Read->Close for each read request.
            // NCL is OK with extra calls - see SIGLO-30099

            if (!readStreamOpen)
            {
                if ((result = nn::fs::OpenFile(
                                    &fileHandle,
                                    m_Name,
                                    nn::fs::OpenMode_Read)).IsFailure())
                {
                    return result;
                }
                readStreamOpen = true;
            }

            if ((result = nn::fs::ReadFile(
                                    &fsReadCount,
                                    fileHandle,
                                    m_FilePosition,
                                    m_pBuffer,
                                    m_BufferSize)).IsFailure())
            {
                break;
            }

            m_BufferPosition = 0;
            m_FilePosition  += static_cast<uint32_t>(fsReadCount);

            if ((m_BufferCount = static_cast<uint32_t>(fsReadCount)) == 0)
            {
                break;
            }
        }
    }

    *pReadLength = readCount;

    if (readStreamOpen)
    {
        nn::fs::CloseFile(fileHandle);
    }

    return result;
}

nn::Result Stream::Flush()
NN_NOEXCEPT
{
    nn::Result result;

    if (m_BufferCount)
    {
        if ((result = nn::fs::WriteFile(
                                m_FileHandle,
                                m_FilePosition,
                                m_pBuffer,
                                m_BufferCount,
                                nn::fs::WriteOption::MakeValue(0))
                                ).IsFailure())
        {
            return result;
        }

        m_FilePosition += m_BufferCount;
        m_BufferCount   = 0;
    }

    return result;
}

nn::Result Stream::WriteStream(const uint8_t* pBuffer, uint32_t length)
NN_NOEXCEPT
{
    nn::Result result;
    uint32_t   count;

    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    if (!m_Initialized || m_StreamMode != StreamMode_Write)
    {
        return nn::erpt::ResultNotInitialized();
    }

    if (pBuffer == nullptr && length > 0)
    {
        return nn::erpt::ResultInvalidArgument();
    }

    while (length > 0)
    {
        if ((count = Min(m_BufferSize - m_BufferCount, length)) > 0)
        {
            std::memcpy(m_pBuffer + m_BufferCount, pBuffer, count);
            m_BufferCount += count;
            length        -= count;
            pBuffer       += count;
        }

        if (m_BufferSize - m_BufferCount == 0)
        {
            if ((result = Flush()).IsFailure())
            {
                return result;
            }
        }
    }

    return ResultSuccess();
}

nn::Result Stream::OpenStream(const char* path, StreamMode mode, uint32_t streamBufferSize)
NN_NOEXCEPT
{
    nn::Result result;

    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    if (m_Initialized)
    {
        return nn::erpt::ResultAlreadyInitialized();
    }

    if (mode == StreamMode_Write)
    {
        // Limitation imposed by FS team: OpenFile() can't create files.
        // Workaround: check result code and try calling CreateFile().

        while ((result = nn::fs::OpenFile(
                                &m_FileHandle,
                                path,
                                nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
                                ).IsFailure())
        {
            if (result <= nn::fs::ResultPathNotFound() &&
               (result  = nn::fs::CreateFile(path, 0)).IsSuccess())
            {
                continue;
            }
            return result;
        }
        nn::fs::SetFileSize(m_FileHandle, 0);
        std::strncpy(m_Name, path, sizeof(m_Name));
    }
    else if (mode == StreamMode_Read)
    {
        std::strncpy(m_Name, path, sizeof(m_Name));
    }
    else
    {
        return nn::erpt::ResultInvalidArgument();
    }

    if ((m_pBuffer = reinterpret_cast<uint8_t*>(
                            nn::lmem::AllocateFromExpHeap(
                                g_HeapHandle,
                                streamBufferSize))) == nullptr)
    {
        nn::fs::CloseFile(m_FileHandle);
        return nn::erpt::ResultOutOfMemory();
    }

    m_BufferSize     = streamBufferSize;
    m_BufferCount    = 0;
    m_FilePosition   = 0;
    m_BufferPosition = 0;
    m_StreamMode     = mode;
    m_Initialized    = true;

    return ResultSuccess();
}

void Stream::CloseStream()
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        if (m_FsAvailable && m_StreamMode == StreamMode_Write)
        {
            Flush();
            nn::fs::FlushFile(m_FileHandle);
            nn::fs::CloseFile(m_FileHandle);
        }
        nn::lmem::FreeToExpHeap(g_HeapHandle, m_pBuffer);
        m_Initialized = false;
    }
}

nn::Result Stream::DeleteStream(const char* pPath)
NN_NOEXCEPT
{
    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    return nn::fs::DeleteFile(pPath);
}

nn::Result Stream::CommitStream()
NN_NOEXCEPT
{
    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    #if !defined(USE_HOSTFS)
    nn::fs::CommitSaveData(ReportStoragePath);
    #endif

    return ResultSuccess();
}

nn::Result Stream::GetStreamSize(int64_t* pSize, const char* path)
NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;

    if (!m_FsAvailable)
    {
        return nn::erpt::ResultPowerStateViolation();
    }

    if ((result = nn::fs::OpenFile(
                    &fileHandle,
                    path,
                    nn::fs::OpenMode_Read)
                    ).IsSuccess())
    {
        result = nn::fs::GetFileSize(pSize, fileHandle);
        nn::fs::CloseFile(fileHandle);
    }

    return result;
}

nn::Result Stream::GetStreamSize(int64_t* pSize)
NN_NOEXCEPT
{
    return GetStreamSize(pSize, m_Name);
}

Stream::Stream()
NN_NOEXCEPT :
    m_BufferSize(0),
    m_FilePosition(0),
    m_BufferCount(0),
    m_pBuffer(nullptr),
    m_StreamMode(StreamMode_Invalid),
    m_Initialized(false)
{

}

Stream::~Stream()
NN_NOEXCEPT
{
    CloseStream();
}

}}}
