﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/time.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>

#include "erptsrv_Context.h"
#include "erptsrv_ContextRecord.h"
#include "erptsrv_Journal.h"
#include "erptsrv_Service.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::lmem::HeapHandle     g_HeapHandle;
nn::sf::ExpHeapAllocator g_ServiceFrameworkAllocator;

void* Allocate(size_t size)
NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void Deallocate(void* p, size_t size)
NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

#if !defined(USE_HOSTFS)
const nn::fs::SystemSaveDataId SystemSaveDataId = 0x80000000000000d1;
const uint32_t SystemSaveDataFlags  = nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData;
const int SystemSaveDataJournalSize = 160 * 1024;
const int SystemSaveDataSize        =   6 * 1024 * 1024;

nn::Result ExtendSystemSaveData()
NN_NOEXCEPT
{
    int64_t currentJournalSize;
    int64_t currentSaveDataSize;

    // Get the save data size and the journal size if those are available. Otherwise, return.
    NN_RESULT_DO(nn::fs::GetSaveDataJournalSize(&currentJournalSize, SystemSaveDataId));
    NN_RESULT_DO(nn::fs::GetSaveDataAvailableSize(&currentSaveDataSize, SystemSaveDataId));

    if (currentJournalSize < SystemSaveDataJournalSize || currentSaveDataSize < SystemSaveDataSize)
    {
        NN_RESULT_DO(nn::fs::ExtendSaveData(
            nn::fs::SaveDataSpaceId::System,
            SystemSaveDataId,
            SystemSaveDataSize,
            SystemSaveDataJournalSize));
    }
    NN_RESULT_SUCCESS;
}

nn::Result MountSystemSaveData()
NN_NOEXCEPT
{
    nn::fs::DisableAutoSaveDataCreation();

    // SIGLO-77668: Extend the save data size and the journal size if those are available and small.
    ExtendSystemSaveData();

    NN_RESULT_TRY(nn::fs::MountSystemSaveData(ReportStoragePath, SystemSaveDataId))
    NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateSystemSaveData(SystemSaveDataId, SystemSaveDataSize, SystemSaveDataJournalSize, SystemSaveDataFlags));
        NN_RESULT_DO(nn::fs::MountSystemSaveData(ReportStoragePath, SystemSaveDataId));
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}
#endif

nn::Result Initialize(uint8_t* pMemoryIn, size_t memorySize)
NN_NOEXCEPT
{
    nn::Result result;

    result = nn::time::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    g_HeapHandle = nn::lmem::CreateExpHeap(
                    pMemoryIn,
                    memorySize,
                    nn::lmem::CreationOption_ThreadSafe);

    NN_ABORT_UNLESS(g_HeapHandle != nullptr);

    nn::fs::SetAllocator(Allocate, Deallocate);

    #if defined(USE_HOSTFS)
    result = nn::fs::MountHost(ReportStoragePath, "c:/temp");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    #else
    result = MountSystemSaveData();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    #endif

    g_ServiceFrameworkAllocator.Attach(g_HeapHandle);

    if (result.IsSuccess())
    {
        for (auto i = 0; i < CategoryId_Last; i++)
        {
            Context *pContext = new Context(static_cast<CategoryId>(i), CategoryLogDepth[i]);
            NN_ABORT_UNLESS(pContext != nullptr);
        }

        Journal::Restore();
    }

    return result;
}

nn::Result InitializeAndStartService()
NN_NOEXCEPT
{
    return InitializeService();
}

nn::Result SetSerialNumberAndOsVersion(
    const char* pSerialNumber,     uint32_t serialNumberLength,
    const char* pOsVersion,        uint32_t osVersionLength,
    const char* pPrivateOsVersion, uint32_t privateOsVersionLength)
NN_NOEXCEPT
{
    return Context::SetSerialNumberAndOsVersion(
                pSerialNumber,      serialNumberLength,
                pOsVersion,         osVersionLength,
                pPrivateOsVersion,  privateOsVersionLength);
}

nn::Result SetProductModel(
    const char* pProductModel, uint32_t productModelLength
)
NN_NOEXCEPT
{
    auto pRecord = new ContextRecord(erpt::CategoryId::ProductModelInfo);
    NN_RESULT_DO(pRecord->Add(erpt::FieldId::ProductModel, pProductModel, productModelLength));
    NN_RESULT_DO(Context::SubmitContext(pRecord));
    NN_RESULT_SUCCESS;
}

nn::Result SetRegionSetting(
    const char* pRegion, uint32_t regionLength
)
NN_NOEXCEPT
{
    auto pRecord = new ContextRecord(erpt::CategoryId::RegionSettingInfo);
    NN_RESULT_DO(pRecord->Add(erpt::FieldId::RegionSetting, pRegion, regionLength));
    NN_RESULT_DO(Context::SubmitContext(pRecord));
    NN_RESULT_SUCCESS;
}

void Wait()
NN_NOEXCEPT
{
    WaitService();
}

}}}
