﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_Endian.h>
#include <nn/util/util_StringUtil.h>
#include <nn/erpt/erpt_TypesPrivate.h>
#include <nn/erpt/server/erpt_ServerTypes.h>

#include "erptsrv_Report.h"

namespace nn   {
namespace erpt {
namespace srv  {

// Formats data into msgpack compliant stream
// Only the necessary minimum subset of data types
// is supported.

class Formatter
{
private:

    enum ElementCount
    {
        ElementCount_16     = 16,
        ElementCount_32     = 32,
        ElementCount_256    = 256,
        ElementCount_16384  = 16384,
    };

    static inline ValueTypeTag GetTag(int8_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::I8;
    }

    static inline ValueTypeTag GetTag(int16_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::I16;
    }

    static inline ValueTypeTag GetTag(int32_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::I32;
    }

    static inline ValueTypeTag GetTag(int64_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::I64;
    }

    static inline ValueTypeTag GetTag(uint8_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::U8;
    }

    static inline ValueTypeTag GetTag(uint16_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::U16;
    }

    static inline ValueTypeTag GetTag(uint32_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::U32;
    }

    static inline ValueTypeTag GetTag(uint64_t dummy)
    NN_NOEXCEPT
    {
        NN_UNUSED(dummy);
        return ValueTypeTag::U64;
    }

    static nn::Result AddId(Report* pReport, FieldId id)
    NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(MaxFieldStringSize < ElementCount_256);

        nn::Result result;
        uint32_t   fieldIdLength = static_cast<uint32_t>(nn::util::Strnlen(FieldString[id], MaxFieldStringSize));

        if (fieldIdLength < ElementCount_32)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Str | fieldIdLength))).IsFailure() ||
                (result = pReport->Write(FieldString[id], fieldIdLength)).IsFailure())
            {
                return result;
            }
        }
        else
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Str256))).IsFailure() ||
                (result = pReport->Write(static_cast<uint8_t>(fieldIdLength))).IsFailure() ||
                (result = pReport->Write(FieldString[id], fieldIdLength)).IsFailure())
            {
                return result;
            }
        }

        return ResultSuccess();
    }

    template <typename T>
    static nn::Result AddValue(Report* pReport, T value)
    NN_NOEXCEPT
    {
        nn::Result result;
        T          data;
        uint8_t    tag = static_cast<uint8_t>(GetTag(value));

        nn::util::StoreBigEndian(&data, value);

        if ((result = pReport->Write(tag)).IsFailure() ||
            (result = pReport->Write(reinterpret_cast<uint8_t*>(&data), sizeof(data))).IsFailure())
        {
            return result;
        }

        return ResultSuccess();
    }

    template <typename T>
    static nn::Result AddValueArray(Report* pReport, T * array, uint32_t count)
    NN_NOEXCEPT
    {
        nn::Result result;
        uint16_t   data;

        if (count < ElementCount_16)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Array16 | count))).IsFailure())
            {
                return result;
            }
        }
        else if (count < ElementCount_16384)
        {
            nn::util::StoreBigEndian(&data, static_cast<uint16_t>(count));

            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Array16384))).IsFailure() ||
                (result = pReport->Write(data)).IsFailure())
            {
                return result;
            }
        }
        else
        {
            return nn::erpt::ResultFormatterError();
        }

        for (uint32_t i = 0; i < count; i++)
        {
            if ((result = AddValue(pReport, array[i])).IsFailure())
            {
                return result;
            }
        }

        return ResultSuccess();
    }

    template <typename T>
    static nn::Result AddIdValuePair(Report* pReport, FieldId id, T value)
    NN_NOEXCEPT
    {
        nn::Result result;

        if ((result = AddId(pReport, id)).IsFailure() ||
            (result = AddValue(pReport, value)).IsFailure())
        {
            return result;
        }

        return ResultSuccess();
    }

    template <typename T>
    static nn::Result AddIdValueArray(Report* pReport, FieldId id, T * array, uint32_t count)
    NN_NOEXCEPT
    {
        nn::Result result;

        if ((result = AddId(pReport, id)).IsFailure() ||
            (result = AddValueArray(pReport, array, count)).IsFailure())
        {
            return result;
        }

        return ResultSuccess();
    }

public:
    template <typename T>
    static nn::Result AddField(Report* pReport, FieldId id, T value)
    NN_NOEXCEPT
    {
        return AddIdValuePair<T>(pReport, id, value);
    }

    template <typename T>
    static nn::Result AddField(Report* pReport, FieldId id, T* array, uint32_t count)
    NN_NOEXCEPT
    {
        return AddIdValueArray<T>(pReport, id, array, count);
    }

    static nn::Result AddField(Report* pReport, FieldId id, bool value)
    NN_NOEXCEPT
    {
        nn::Result result;

        if ((result = AddId(pReport, id)).IsFailure() ||
            (result = pReport->Write(static_cast<uint8_t>(value ? ValueTypeTag::True : ValueTypeTag::False))).IsFailure())
        {
            return result;
        }

        return ResultSuccess();
    }

    static nn::Result AddField(Report* pReport, FieldId id, char* string, uint32_t length)
    NN_NOEXCEPT
    {
        nn::Result result;
        uint32_t   stringLength;
        uint16_t   data;

        if ((result = AddId(pReport, id)).IsFailure())
        {
            return result;
        }

        // NSD asked to remove terminating character from strings
        stringLength = string != nullptr ?
                       static_cast<uint32_t>(nn::util::Strnlen(string, length)) :
                       0;

        if (stringLength < ElementCount_32)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Str | stringLength))).IsFailure() ||
                (result = pReport->Write(string, stringLength)).IsFailure())
            {
                return result;
            }
        }
        else if (stringLength < ElementCount_256)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Str256))).IsFailure() ||
                (result = pReport->Write(static_cast<uint8_t>(stringLength))).IsFailure() ||
                (result = pReport->Write(string, stringLength)).IsFailure())
            {
                return result;
            }
        }
        else if (stringLength < ElementCount_16384)
        {
            nn::util::StoreBigEndian(&data, static_cast<uint16_t>(stringLength));
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Str16384))).IsFailure() ||
                (result = pReport->Write(data)).IsFailure() ||
                (result = pReport->Write(string, stringLength)).IsFailure())
            {
                return result;
            }
        }
        else
        {
            return nn::erpt::ResultFormatterError();
        }

        return ResultSuccess();
    }

    static nn::Result AddField(Report* pReport, FieldId id, uint8_t* array, uint32_t count)
    NN_NOEXCEPT
    {
        nn::Result result;
        uint16_t   data;

        if ((result = AddId(pReport, id)).IsFailure())
        {
            return result;
        }

        if (count < ElementCount_256)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Bin))).IsFailure() ||
                (result = pReport->Write(static_cast<uint8_t>(count))).IsFailure() ||
                (result = pReport->Write(array, count)).IsFailure())
            {
                return result;
            }
        }
        else if (count < ElementCount_16384)
        {
            nn::util::StoreBigEndian(&data, static_cast<uint16_t>(count));
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Bin16384))).IsFailure() ||
                (result = pReport->Write(data)).IsFailure() ||
                (result = pReport->Write(array, count)).IsFailure())
            {
                return result;
            }
        }
        else
        {
            return nn::erpt::ResultFormatterError();
        }

        return ResultSuccess();
    }

    static nn::Result Begin(Report* pReport, uint32_t recordCount)
    NN_NOEXCEPT
    {
        nn::Result result;
        uint16_t   data;

        if (recordCount < ElementCount_16)
        {
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Map16 | recordCount))).IsFailure())
            {
                return result;
            }
        }
        else if (recordCount < ElementCount_16384)
        {
            nn::util::StoreBigEndian(&data, static_cast<uint16_t>(recordCount));
            if ((result = pReport->Write(static_cast<uint8_t>(ValueTypeTag::Map16384))).IsFailure() ||
                (result = pReport->Write(data)).IsFailure())
            {
                return result;
            }
        }
        else
        {
            return nn::erpt::ResultFormatterError();
        }

        return ResultSuccess();
    }

    static nn::Result End(Report* pReport)
    NN_NOEXCEPT
    {
        NN_UNUSED(pReport);
        return ResultSuccess();
    }
};

}}}
