﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>

#include "erptsrv_ContextRecord.h"
#include "erptsrv_Context.h"

namespace nn   {
namespace erpt {
namespace srv  {

uint32_t ContextRecord::m_ContextRecordCount = 0;

nn::Result ContextRecord::Initialize(
    const ContextEntry* pContextIn,
    const uint8_t*      pDataIn,
    uint32_t            dataLength)
NN_NOEXCEPT
{
    nn::Result result;

    if (dataLength > MaxArrayBufferLength)
    {
        return nn::erpt::ResultInvalidArgument();
    }

    m_Context.version         = pContextIn->version;
    m_Context.fieldCount      = pContextIn->fieldCount;
    m_Context.category        = pContextIn->category;
    m_Context.arrayFreeCount  = 0;
    m_Context.arrayBufferSize = dataLength;
    m_Context.pArrayBuffer    = nullptr;

    if (m_Context.fieldCount > FieldsPerContext ||
        m_Context.category   < 0 ||
        m_Context.category  >= CategoryId_Last)
    {
        m_Context.fieldCount = 0;
        return nn::erpt::ResultInvalidArgument();
    }

    for (uint32_t i = 0; i < m_Context.fieldCount; i++)
    {
        m_Context.fields[i] = pContextIn->fields[i];

        if (m_Context.fields[i].id   <  0 ||
            m_Context.fields[i].id   >= FieldId_Last ||
            m_Context.fields[i].type <  0 ||
            m_Context.fields[i].type >= FieldType_Last)
        {
            m_Context.fieldCount = 0;
            return nn::erpt::ResultInvalidArgument();
        }

        if (FieldToTypeMap[m_Context.fields[i].id] != m_Context.fields[i].type)
        {
            NN_SDK_LOG("[erpt]: field '%s' should be of type '%s' (currently '%s')\n",
                       FieldString[m_Context.fields[i].id],
                       TypeString[FieldToTypeMap[m_Context.fields[i].id]],
                       TypeString[m_Context.fields[i].type]);
            m_Context.fieldCount = 0;
            return nn::erpt::ResultFieldTypeMismatch();
        }

        if (FieldToCategoryMap[m_Context.fields[i].id] != m_Context.category)
        {
            NN_SDK_LOG("[erpt]: field '%s' can't be used with category '%s', "
                       "requires category '%s'\n",
                       FieldString[m_Context.fields[i].id],
                       CategoryString[m_Context.category],
                       CategoryString[FieldToCategoryMap[m_Context.fields[i].id]]);
            m_Context.fieldCount = 0;
            return nn::erpt::ResultFieldCategoryMismatch();
        }

        if (m_Context.fields[i].type == FieldType_String   ||
            m_Context.fields[i].type == FieldType_U8Array  ||
            m_Context.fields[i].type == FieldType_U32Array ||
            m_Context.fields[i].type == FieldType_U64Array ||
            m_Context.fields[i].type == FieldType_I32Array ||
            m_Context.fields[i].type == FieldType_I64Array)
        {
            if (m_Context.fields[i].u.arrayField.startIndex > dataLength ||
                m_Context.fields[i].u.arrayField.size > dataLength ||
                m_Context.fields[i].u.arrayField.startIndex +
                m_Context.fields[i].u.arrayField.size > dataLength)
            {
                m_Context.fieldCount = 0;
                return nn::erpt::ResultInvalidArgument();
            }

            if( m_Context.fields[i].u.arrayField.size > nn::erpt::MaxArrayFieldSize )
            {
                NN_SDK_LOG("[erpt]: field '%s'(%u) is too large, it must be <= nn::erpt::MaxArrayFieldSize(%u)\n",
                            FieldString[m_Context.fields[i].id], m_Context.fields[i].u.arrayField.size, nn::erpt::MaxArrayFieldSize);
                m_Context.fieldCount = 0;
                return nn::erpt::ResultArrayFieldSizeTooLarge();
            }
        }
    }

    if (dataLength > 0)
    {
        m_Context.pArrayBuffer = reinterpret_cast<uint8_t*>(
                lmem::AllocateFromExpHeap(
                    g_HeapHandle,
                    dataLength,
                    sizeof(uint64_t))
                );

        if (m_Context.pArrayBuffer == nullptr)
        {
            NN_SDK_LOG("[erpt]: failed to allocate %u bytes for category '%s', allocatable size is %zu bytes, total free size is %zu bytes\n",
                dataLength, CategoryString[m_Context.category],
                lmem::GetExpHeapAllocatableSize(g_HeapHandle, sizeof(uint64_t)),
                lmem::GetExpHeapTotalFreeSize(g_HeapHandle));
            m_Context.fieldCount = 0;
            return nn::erpt::ResultOutOfMemory();
        }

        std::memcpy(m_Context.pArrayBuffer, pDataIn, dataLength);
    }

    m_ContextRecordCount += m_Context.fieldCount;

    return ResultSuccess();
}

uint32_t ContextRecord::GetRecordCount()
NN_NOEXCEPT
{
    return m_ContextRecordCount;
}

nn::Result ContextRecord::Add(FieldId id, uint64_t valueU64)
NN_NOEXCEPT
{
    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU64;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU64 = valueU64;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    return ResultSuccess();
}

nn::Result ContextRecord::Add(FieldId id, bool valueBool)
NN_NOEXCEPT
{
    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_Bool;
    m_Context.fields[m_Context.fieldCount].u.boolField = valueBool;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    return ResultSuccess();
}

nn::Result ContextRecord::Add(FieldId id, uint32_t valueU32)
NN_NOEXCEPT
{
    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU32;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU32 = valueU32;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    return ResultSuccess();
}

nn::Result ContextRecord::Add(FieldId id, int64_t valueI64)
NN_NOEXCEPT
{
    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI64;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI64 = valueI64;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    return ResultSuccess();
}

nn::Result ContextRecord::Add(FieldId id, int32_t valueI32)
NN_NOEXCEPT
{
    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI32;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI32 = valueI32;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    return ResultSuccess();
}

nn::Result ContextRecord::Add(FieldId id, const char* pBuffer, uint32_t bufferLength)
NN_NOEXCEPT
{
    uint32_t startIndex;

    if (m_Context.fieldCount >= FieldsPerContext)
    {
        return nn::erpt::ResultOutOfFieldSpace();
    }

    if (bufferLength > m_Context.arrayFreeCount)
    {
        return nn::erpt::ResultOutOfArraySpace();
    }

    startIndex = m_Context.arrayBufferSize - m_Context.arrayFreeCount;

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_String;
    m_Context.fields[m_Context.fieldCount].u.arrayField.size = bufferLength;
    m_Context.fields[m_Context.fieldCount].u.arrayField.startIndex = startIndex;
    m_Context.arrayFreeCount -= bufferLength;
    m_Context.fieldCount++;
    m_ContextRecordCount++;

    std::memcpy(m_Context.pArrayBuffer + startIndex, pBuffer, bufferLength);

    return ResultSuccess();
}

ContextRecord::ContextRecord()
NN_NOEXCEPT
{
    std::memset(&m_Context, 0x0, sizeof(m_Context));
}

ContextRecord::ContextRecord(CategoryId category)
NN_NOEXCEPT
{
    m_Context.version         = ERRVERSION;
    m_Context.fieldCount      = 0;
    m_Context.category        = category;
    m_Context.arrayBufferSize = 0;
    m_Context.arrayFreeCount  = 0;
    m_Context.pArrayBuffer    = reinterpret_cast<uint8_t*>(
                                lmem::AllocateFromExpHeap(
                                    g_HeapHandle,
                                    ArrayBufferLength,
                                    sizeof(uint64_t))
                                );

    if (m_Context.pArrayBuffer != nullptr)
    {
        m_Context.arrayBufferSize = ArrayBufferLength;
        m_Context.arrayFreeCount  = ArrayBufferLength;
    }

    return;
}

ContextRecord::~ContextRecord()
NN_NOEXCEPT
{
    if (m_Context.pArrayBuffer)
    {
        lmem::FreeToExpHeap(g_HeapHandle, m_Context.pArrayBuffer);
    }

    NN_ABORT_UNLESS(m_ContextRecordCount >= m_Context.fieldCount);
    m_ContextRecordCount -= m_Context.fieldCount;
}

}}}
