﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>

#include "eclct_Pcie.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {


Result InitializePcie() NN_NOEXCEPT
{
    Result result;

    result = nn::pcie::InitializeLoggedState();

    return result;
}

Result GetPcieLoggedStateEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    Result result;

    result = nn::pcie::GetLoggedStateEvent(pOutSystemEvent, eventClearMode, true);

    return result;
}

Result UpdatePcieLoggedState() NN_NOEXCEPT
{
    Result result;
    nn::pcie::LoggedState loggedState;
    bool wasLoggedErrorPending = false;

    if((result = nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, false)).IsSuccess())
    {
        uint32_t portStatistics[7];
        uint32_t val32 = 0;
        nn::erpt::Context context(nn::erpt::PcieLoggedStateInfo);

        // Add port0 parameters
        val32 = 0;
        for(int bit = 0; bit < 32; bit++)
        {
            val32 |= (loggedState.rootComplex.ports[0].flagSet[bit]) ? (1 << bit) : 0;
        }
        context.Add(nn::erpt::PciePort0Flags,         val32);
        context.Add(nn::erpt::PciePort0Speed,         static_cast<uint8_t>(loggedState.rootComplex.ports[0].speed));
        context.Add(nn::erpt::PciePort0ResetTimeInUs, loggedState.rootComplex.ports[0].portResetTimeInUs);
        context.Add(nn::erpt::PciePort0IrqCount,      loggedState.rootComplex.ports[0].irqCount);

        // Add port0 statistics
        portStatistics[0] = loggedState.rootComplex.ports[0].afiMsgPmeCount;
        portStatistics[1] = loggedState.rootComplex.ports[0].afiMsgIntxCount;
        portStatistics[2] = loggedState.rootComplex.ports[0].afiMsgPcieFatalErrorCount;
        portStatistics[3] = loggedState.rootComplex.ports[0].afiMsgPcieNonFatalErrorCount;
        portStatistics[4] = loggedState.rootComplex.ports[0].afiMsgPcieCorrectableErrorCount;
        portStatistics[5] = loggedState.rootComplex.ports[0].afiMsgPcieRootPortIntCount;
        portStatistics[6] = loggedState.rootComplex.ports[0].afiMsgHotplugCount;
        context.Add(nn::erpt::PciePort0Statistics, portStatistics, sizeof(portStatistics) / sizeof(portStatistics[0]));

        // Add port1 parameters
        val32 = 0;
        for(int bit = 0; bit < 32; bit++)
        {
            val32 |= (loggedState.rootComplex.ports[1].flagSet[bit]) ? (1 << bit) : 0;
        }
        context.Add(nn::erpt::PciePort1Flags,         val32);
        context.Add(nn::erpt::PciePort1Speed,         static_cast<uint8_t>(loggedState.rootComplex.ports[1].speed));
        context.Add(nn::erpt::PciePort1ResetTimeInUs, loggedState.rootComplex.ports[1].portResetTimeInUs);
        context.Add(nn::erpt::PciePort1IrqCount,      loggedState.rootComplex.ports[1].irqCount);

        // Add port1 statistics
        portStatistics[0] = loggedState.rootComplex.ports[1].afiMsgPmeCount;
        portStatistics[1] = loggedState.rootComplex.ports[1].afiMsgIntxCount;
        portStatistics[2] = loggedState.rootComplex.ports[1].afiMsgPcieFatalErrorCount;
        portStatistics[3] = loggedState.rootComplex.ports[1].afiMsgPcieNonFatalErrorCount;
        portStatistics[4] = loggedState.rootComplex.ports[1].afiMsgPcieCorrectableErrorCount;
        portStatistics[5] = loggedState.rootComplex.ports[1].afiMsgPcieRootPortIntCount;
        portStatistics[6] = loggedState.rootComplex.ports[1].afiMsgHotplugCount;
        context.Add(nn::erpt::PciePort1Statistics, portStatistics, sizeof(portStatistics) / sizeof(portStatistics[0]));

        // Add function0 statistics
        context.Add(nn::erpt::PcieFunction0VendorId,   loggedState.functions[0].vendorId);
        context.Add(nn::erpt::PcieFunction0DeviceId,   loggedState.functions[0].deviceId);
        context.Add(nn::erpt::PcieFunction0PmState,    static_cast<uint8_t>(loggedState.functions[0].pmState));
        context.Add(nn::erpt::PcieFunction0IsAcquired, loggedState.functions[0].isAcquired);

        // Add function1 statistics
        context.Add(nn::erpt::PcieFunction1VendorId,   loggedState.functions[1].vendorId);
        context.Add(nn::erpt::PcieFunction1DeviceId,   loggedState.functions[1].deviceId);
        context.Add(nn::erpt::PcieFunction1PmState,    static_cast<uint8_t>(loggedState.functions[1].pmState));
        context.Add(nn::erpt::PcieFunction1IsAcquired, loggedState.functions[1].isAcquired);

        // Add global root complex statistics
        context.Add(nn::erpt::PcieGlobalRootComplexStatistics,
                    reinterpret_cast<uint32_t*>(&loggedState.rootComplex.globalCounts),
                    sizeof(loggedState.rootComplex.globalCounts)/
                    sizeof(loggedState.rootComplex.globalCounts.afiInitiatorSlaveReadErrorCount));

        // Add PLL resistor calibration
        context.Add(nn::erpt::PciePllResistorCalibrationValue, loggedState.rootComplex.pllResistorCalibrationValue);

        // Submit context
        context.SubmitContext();
    }

    return result;
}


}}
