﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include <nn/dns/parser.h>

//#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_MODULE_NAME "Question" // NOLINT(preprocessor/const)
#include "dns_ParserLog.h"
#include "dns_ParserMacros.h"

extern "C"
{
#include <nnc/dns/parser.h>
};

NN_DNSPARSER_STATIC_ASSERT(sizeof(struct nndnsparserQuestion) == sizeof(nn::dns::parser::Question));
NN_DNSPARSER_STATIC_ASSERT(alignof(struct nndnsparserQuestion) == alignof(nn::dns::parser::Question));

namespace nn { namespace dns { namespace parser {

Question::Question() :
    m_pMessage(nullptr),
    m_DirtySize(0),
    m_Name(Label()),
    m_Type {},
    m_Class {}
{
    m_Range.pStart = m_Range.pEnd = nullptr;
};

Question::Question(const Question& rhs) :
    m_pMessage(rhs.m_pMessage),
    m_Range(rhs.m_Range),
    m_DirtySize(rhs.m_DirtySize),
    m_Name(rhs.m_Name),
    m_Type(rhs.m_Type),
    m_Class(rhs.m_Class)
{
};

Question::~Question()
{
};

Question& Question::operator=(const Question& rhs)
{
    if (this == &rhs)
    {
        goto bail;
    };

    m_pMessage = rhs.m_pMessage;
    m_Range = rhs.m_Range;
    m_DirtySize = rhs.m_DirtySize;
    m_Name = rhs.m_Name;
    m_Type = rhs.m_Type;
    m_Class = rhs.m_Class;

bail:
    return *this;
};

const Message* & Question::GetMessage() NN_NOEXCEPT
{
    return m_pMessage;
};

Label& Question::GetName() NN_NOEXCEPT
{
    return m_Name;
};

TypeConstant& Question::GetType() NN_NOEXCEPT
{
    return m_Type;
};

ClassConstant& Question::GetClass() NN_NOEXCEPT
{
    return m_Class;
};

void Question::Initialize(const Message* pMessage) NN_NOEXCEPT
{
    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, pMessage) ==
                               offsetof(nn::dns::parser::Question, m_pMessage));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, range) ==
                               offsetof(nn::dns::parser::Question, m_Range));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, dirtySize) ==
                               offsetof(nn::dns::parser::Question, m_DirtySize));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, name) ==
                               offsetof(nn::dns::parser::Question, m_Name));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, type) ==
                               offsetof(nn::dns::parser::Question, m_Type));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestion, qclass) ==
                               offsetof(nn::dns::parser::Question, m_Class));

    memset(this, 0, sizeof(*this));
    m_pMessage = pMessage;
    m_Name.Initialize(pMessage);

    return;
};

ssize_t Question::SizeOf() const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    ssize_t rc = -1;

    if ( static_cast<unsigned int>(LibraryConstant::ObjectIsDirty) != (rc = m_DirtySize ))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if ( -1 == (rc = m_Name.SizeOf()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    rc += sizeof(m_Type);
    rc += sizeof(m_Class);

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

bool Question::operator==(const Question& that) const NN_NOEXCEPT
{
    bool rc = false;

    if (this == &that)
    {
        rc = true;
        goto bail;
    }
    else if (false == (m_Name == that.m_Name))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Type != that.m_Type)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Class != that.m_Class)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    rc = true;

bail:
    return rc;
};

ssize_t Question::FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    ssize_t rc = -1;
    const uint8_t* pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);

    m_Range.pStart = pCursor;

    if (-1 == (rc = m_Name.FromBuffer(pCursor, size)))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };
    size -= rc;
    pCursor += rc;

    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Type,  pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Class, pCursor, size, nn::socket::InetNtohs, bail);

    rc = m_DirtySize = pCursor - pBuffer;
    m_Range.pEnd = pCursor;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Question::ToBuffer(uint8_t * const pBuffer, size_t size) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    ssize_t rc = -1;
    uint8_t* pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);

    if (-1 == ( rc = m_Name.ToBuffer(pCursor, size)))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };
    size -= rc;
    pCursor += rc;

    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_Type, bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_Class, bail);

    rc = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

}}}; //nn::dnsparser
