﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief This file contains utility functions used throughout the project.
 */

namespace nn { namespace dhcps { namespace detail {

/**
 * @brief This function returns a mac address string formatted in
 * "aa:bb:cc:dd:ee:ff" notation.
 *
 * @param[out] pDest This parameter is a pointer to a char array that will
 * contain the formatted string on success.
 *
 * @param[in] size This parameter is the size of dst in memory.
 *
 * @param[in] mac This parameter is an integer that contains the mac
 * address as converted with @ref BytesToInteger.
 *
 * @return This function returns dst with a string on success
 * otherwise it returns nullptr on error.
 */
const char* MacAddressToString(char* pDest, size_t size, const EthernetMacAddress mac) NN_NOEXCEPT;

/**
 * @brief This function returns a 32-bit value that represents the
 * current time in seconds.
 *
 * @detail It is intended to be used with DHCP time functions which
 * are in 32-bit and not 64-bit function.
 *
 * @return The current time in seconds.
 */
uint32_t Time() NN_NOEXCEPT;

/**
 * @brief Get the relative time between now and then.
 *
 * @param[in] now The 'current' time to base the relative time on.
 *
 * @param[in] then The other time.
 *
 * @details
 * - If then > now it returns the difference.
 * - Otherwise it returns zero.
 *
 * @return the time difference
 */
uint32_t RelativeTime(uint32_t now, uint32_t then) NN_NOEXCEPT;

/**
 * @brief Lock the API lock.
 *
 * @details
 * The API lock is the lock that gates access into the library via
 * the public API.
 */
void ApiLock() NN_NOEXCEPT;

/**
 * @brief Unlock the API lock.
 *
 * @details
 * The API lock is the lock that gates access into the library via
 * the public API.
 */
void ApiUnlock() NN_NOEXCEPT;

/**
 * @brief Lock the Internal lock
 *
 * @details
 * The Internal lock is chiefly used by the DHCP server worker thread
 * to gate access into library internals. If the internal lock is
 * taken then the thread calling the API lock waits until the internal
 * lock is available.
 */
void InternalLock() NN_NOEXCEPT;

/**
 * @brief Unlock the Internal lock;
 *
 * @details
 * The Internal lock is chiefly used by the DHCP server worker thread
 * to gate access into library internals. If the internal lock is
 * taken then the thread calling the API lock waits until the internal
 * lock is available.
 */
void InternalUnlock() NN_NOEXCEPT;

}}}; // nn::dhcps::detail
