﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_UdpSocketManager.h"
#include "dhcps_Coordinator.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "UdpSocketManager"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;

namespace {

int OpenUdpSocket(uint16_t port, uint32_t socketAddress) NN_NOEXCEPT
{
    int rc = -1;
    int burnerSocket = -1;
    int yes = 1;
    nn::socket::Errno errorNumber;
    nn::socket::SockAddrIn serverAddress;

    serverAddress.sin_family = nn::socket::Family::Af_Inet;
    serverAddress.sin_port = nn::socket::InetHtons(port);
    serverAddress.sin_addr.S_addr = socketAddress;

    if (-1 == (burnerSocket = nn::socket::Socket(nn::socket::Family::Af_Inet,
                                                 nn::socket::Type::Sock_Dgram,
                                                 nn::socket::Protocol::IpProto_Udp)))
    {
        errorNumber = nn::socket::GetLastError();
        NN_DETAIL_DHCPS_LOG_MAJOR("Cannot open the socket! %s(%d)\n",
                                       strerror(static_cast<int>(errorNumber)),
                                       errorNumber);
        goto bail;
    }
    else if ( -1 == (rc = nn::socket::SetSockOpt(burnerSocket,
                                                 nn::socket::Level::Sol_Socket,
                                                 nn::socket::Option::So_ReuseAddr,
                                                 &yes,
                                                 sizeof(yes))))
    {
        errorNumber = nn::socket::GetLastError();
        NN_DETAIL_DHCPS_LOG_MAJOR("Unable to setsockopt SOL_SOCKET, So_ReuseAddr; "
                                       "error: %s (%d)\n",
                                       strerror(static_cast<int>(errorNumber)),
                                       errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::Bind(burnerSocket,
                                          reinterpret_cast<nn::socket::SockAddr*>(&serverAddress),
                                          sizeof(serverAddress))))
    {
        errorNumber = nn::socket::GetLastError();
        NN_DETAIL_DHCPS_LOG_MAJOR("Cannot bind a UDP socket to the address!(%s:%d); "
                                       "error: %s (%d).\n",
                                       nn::socket::InetNtoa(serverAddress.sin_addr),
                                       port,
                                       strerror(static_cast<int>(errorNumber)),
                                       errorNumber);
        goto bail;
    };

    rc = burnerSocket;

bail:
    return rc;
};

} // end anonymous namespace

const char* UdpSocketManagerStateToString(UdpSocketManager::State in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::FileOpen);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::Ready);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::FileClosed);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(UdpSocketManager::State::FileError);
    default:
        ;
    };
    return "Unknown UdpSocketManager::State";
};

UdpSocketManager::UdpSocketManager() NN_NOEXCEPT :
m_State(State::Uninitialized),
    m_FileDescriptor(-1),
    m_pCoordinator(nullptr)
{
};

UdpSocketManager::~UdpSocketManager() NN_NOEXCEPT
{
};

int UdpSocketManager::GetFileDescriptor() NN_NOEXCEPT
{
    return m_FileDescriptor;
};

int UdpSocketManager::Initialize(Coordinator* pCoordinator) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("this: %p\n", this);
    int rc = -1;

    if (m_pCoordinator != nullptr)
    {
        NN_SDK_ASSERT(false);
        goto bail;
    };

    m_pCoordinator = pCoordinator;
    ChangeState(State::Initialized);
    rc = 0;

bail:
    return rc;
};

int UdpSocketManager::Finalize() NN_NOEXCEPT
{
    ChangeState(State::Uninitialized);

    if (-1 != m_FileDescriptor)
    {
        nn::socket::Close(m_FileDescriptor);
        m_FileDescriptor = -1;
    };

    m_pCoordinator = nullptr;

    ChangeState(State::Uninitialized);

    return 0;
};

void UdpSocketManager::ChangeState(State in) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("Changing state from: %s (%d) to %s (%d)\n",
                                   UdpSocketManagerStateToString(m_State),
                                   m_State,
                                   UdpSocketManagerStateToString(in),
                                   in);

    SetGlobalState(GlobalState::UdpSocketManager, static_cast<uint8_t>(in));

    m_State = in;
};

void UdpSocketManager::OnEvent(const InternalEvent& e) NN_NOEXCEPT
{
    ssize_t rc = -1;
    switch (e.GetType())
    {
    case EventType::OnFileOpen:
        if (-1 == (m_FileDescriptor = OpenUdpSocket(g_Config.GetDhcpServerUdpPort(),
                                                    g_Config.GetIpAddress().S_addr)))
        {
            ChangeState(State::FileError);
            InternalEvent e(EventType::OnFileError, nullptr, 0);
            m_pCoordinator->OnEvent(e);
        }
        else
        {
            ChangeState(State::FileOpen);
        }
        break;
    case EventType::OnFileRead:
    {
        uint8_t ignore[static_cast<uint32_t>(LibraryConstants::DhcpMaxMtu)];
        if (-1 == (rc = nn::socket::Read(m_FileDescriptor, ignore, sizeof(ignore))))
        {
            ChangeState(State::FileError);
            InternalEvent e(EventType::OnFileError, nullptr, 0);
            m_pCoordinator->OnEvent(e);
        };
        break;
    }
    case EventType::OnFileWrite:
    case EventType::OnFileExcept:
        // ignore
        break;
    case EventType::OnFileClose:
        if (-1 != m_FileDescriptor)
        {
            nn::socket::Close(m_FileDescriptor);
            m_FileDescriptor = -1;
            ChangeState(State::FileClosed);
        };
        break;
    case EventType::OnFileError:
        if (-1 != m_FileDescriptor)
        {
            nn::socket::Close(m_FileDescriptor);
            m_FileDescriptor = -1;
            ChangeState(State::FileClosed);
        };
        break;
    case EventType::OnTimerExpired:
        break;
    default:
        break;
    };

    return;
};

void UdpSocketManager::GetTimeout(nn::socket::TimeVal* pTv) NN_NOEXCEPT
{
    if (pTv == nullptr)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    const unsigned int Timeout = 1;
    pTv->tv_sec = Timeout;
    pTv->tv_usec = 0;
};

}}}; // nn::dhcps::detail
