﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"
#include "dhcps_Constants.h"

/**
 * @file
 *
 * @brief This file defines structures that are used for DHCP
 * message processing.
 */
namespace nn { namespace dhcps { namespace detail {

/**
 * @brief This structure is a representation of the BOOTP message
 */
struct BootpDhcpMessage
{
    /**
     * @brief Message op code / message type.
     * 1 = BOOTREQUEST, 2 = BOOTREPLY
     */
    BootProtocolOperationCode operationCode;

    /**
     * @brief Hardware address type, see ARP section in "Assigned
     * Numbers" RFC; e.g., '1' = 10mb ethernet.
     */
    BootProtocolHardwareType hardwareType;

    /**
     * @brief Hardware address length (e.g.  '6' for 10mb ethernet).
     */
    uint8_t hardwareLength;

    /**
     * @brief Client sets to zero, optionally used by relay agents
     * when booting via a relay agent.
     */
    uint8_t hops;

    /**
     * @brief This field is the transaction ID, a random number chosen
     * by the client, used by the client and server to associate
     * messages and responses between a client and a server.
     */
    uint32_t transactionId;

    /**
     * @brief This field is filled in by client, seconds elapsed since
     * client began address acquisition or renewal process.
     */
    uint16_t seconds;

    /**
     * @brief This field contains bootp message flags.
     *
     * @details Presently only a single bit is used
     *
     *
     * <pre>
     * Flags (see figure 2).
     *                       1 1 1 1 1 1
     *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |B|             MBZ             |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *
     *   B:  BROADCAST flag
     *
     *  MBZ:  MUST BE ZERO (reserved for future use)
     *
     * </pre>
     */
    uint16_t flags;

    /**
     * @brief This field is the client IP address; only filled in if
     * client is in BOUND, RENEW or REBINDING state and can respond
     * to ARP requests.
     */
    nn::socket::InAddr clientIpAddress;

    /**
     * @brief This field is 'your' (i.e. the client's) IP address.
     */
    nn::socket::InAddr yourIpAddress;

    /**
     * @brief This field is the IP address of next server to use in
     * bootstrap; returned in DHCPOFFER, DHCPACK by server.
     */
    nn::socket::InAddr serverIpAddress;

    /**
     * @brief This field is a relay agent IP address that is used in
     * booting via a relay agent.
     */
    nn::socket::InAddr gatewayIpAddress;

    /**
     * @brief This field is the client hardware address.
     * @details It is filled in from the left
     */
    uint8_t clientHardwareAddress[static_cast<size_t>(LibraryConstants::BootpClientHardwareAddressSize)];

    /**
     * @brief This field is the optional server name, a null
     * terminated string.
     */
    char serverName[static_cast<size_t>(LibraryConstants::BootpServerNameSize)];

    /**
     * @brief This is the boot file name, null terminated string;
     *"generic" name or null in DHCPDISCOVER, fully qualified
     * directory-path name in DHCPOFFER.
     */
    char bootFileName[static_cast<size_t>(LibraryConstants::BootpFileNameSize)];

    /**
     * @brief This field is required by DHCP. It extends the bootp
     * specification and is validated against the value of
     * LibraryConstants::DhcpMagicCookie in network byte order to
     * indicate that this bootp message is a DHCP message.
     */
    uint32_t magicCookie;

    /**
     * @brief A flexible array that contains DHCP options. A near-
     * comprehensive list of options are found at @a DhcpOptionCode
     */
    uint8_t options[];

} __packed __aligned(static_cast<int>(LibraryConstants::NetworkStructureAlignment));

/**
 * @brief This structure represents a DHCP option. All options
 * except @a DhcpOptionCode::Pad and @a DhcpOptionCode::End contain type,
 * length, and value(s). Sometimes the values must be
 * further converted.
 */
template <typename T>
struct DhcpOptionContainer
{
    /**
     * @brief This field is the type parameter. For a comprehensive
     * list of option codes see @a DhcpOptionCode
     */
    DhcpOptionCode type;

    /**
     * @brief This field is the length of the value for the option
     * code except DhcpOptionCode::Pad and DhcpOptionCode::End; which
     * are implied as a single byte.
     */
    uint8_t length;

    /**
     * @brief This field contains the DHCP option data.
     */
    T value;
} __packed __aligned(static_cast<int>(LibraryConstants::NetworkStructureAlignment));


}}}; // nn::dhcps::detail
