﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"
#include "dhcps_LeaseTableManager.h"

/**
 * @file
 *
 * @brief This is the header file for the DHCP manager class, which
 * receives DHCP network events, collaborates with the
 * @a LeaseTableManager and creates response events.
 */

namespace nn { namespace dhcps { namespace detail {

class Coordinator;

/**
 * @brief The DHCP Manager class that both receives and sends DHCP
 * network events and allocates leases from the @ref LeaseTableManager.
 */
class DhcpManager : public IEventHandler
{
public:
    /**
     * @brief The states of the DhcpManager class.
     */
    enum class State : uint8_t
    {
        Uninitialized = 0, ///< 0
        Initialized   = 1, ///< 1
        Ready         = 2, ///< 2
        OnDiscover    = 3, ///< 3
        OnDecline     = 4, ///< 4
        OnRequest     = 5, ///< 5
        OnInform      = 6, ///< 6
        OnRelease     = 7, ///< 7
        OnUnhandled   = 8, ///< 8
        Error         = 9, ///< 9
    };

    /**
     * @brief Constructor.
     */
    DhcpManager() NN_NOEXCEPT;

    /**
     * @brief Destructor.
     */
    ~DhcpManager() NN_NOEXCEPT;

    /**
     * @brief Initializes the DhcpManager with the provided
     * @ref Coordinator and @ref LeaseTableManager.
     *
     * @param[in] pCoordinator The provided coordinator.
     *
     * @param[in] pLeaseTableManager The provided LeaseTableManager.
     *
     * @return 0 on success, -1 on error
     */
    int Initialize(Coordinator* pCoordinator, LeaseTableManager* pLeaseTableManager) NN_NOEXCEPT;

    /**
     * @brief Finalizes the DhcpManager.
     *
     * @return 0 on success, -1 on error
     */
    int Finalize() NN_NOEXCEPT;

    /**
     * @brief The @ref IEventHandler event handler.
     *
     * @param[in] e The event to handle.
     */
    virtual void OnEvent(const InternalEvent& e) NN_NOEXCEPT;

    /**
     * @brief Get the desired time for the next timeout event.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT;

private:
    /**
     * @brief The current state.
     */
    State m_State;

    /**
     *  @brief The event @ref Coordinator.
     */
    Coordinator* m_pCoordinator;

    /**
     * @brief The @ref LeaseTableManager.
     */
    LeaseTableManager* m_pLeaseTableManager;

    /**
     * @brief Changes the state variable.
     *
     * @param[in] next The next state to transition to.
     */
    void ChangeState(State next) NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnTimerExpired events.
     */
    void OnTimerExpired() NN_NOEXCEPT;

    /**
     * @brief Called when a new bootp DHCP message is received.
     *
     * @param[in] pStack An array of @ref NetworkLayersContainer
     * that contain the layers of the packet.
     *
     * @param[in] count The number of layers in @a pStack.
     */
    void OnNewMessage(const NetworkLayersContainer pStack[],
                      size_t count) NN_NOEXCEPT;

    /**
     * @brief Event handler for new messages containing a
     * @ref DhcpOptionCode::DhcpMessageType option value of
     * @ref DhcpMessageType::Discover.
     *
     * @param[out] pOutResponse The response object.
     *
     * @param[in] responseSize The size of of the response object in
     * bytes.
     *
     * @param[in] origin The identity of the client that originated the
     * message.
     *
     * @param[in]  pRequest The parsed DHCP bootp request message
     * containing option values.
     *
     * @param[in] requestSize The size of the request.
     *
     * @return
     * - rc > 0: Success. The buffer contains rc bytes as a response.
     * - rc == 0: Success. The buffer contains no bytes.
     * - -1 == 0: Failure
     */
    ssize_t OnDhcpDiscover(BootpDhcpMessage* pOutResponse,
                           size_t responseSize,
                           const DhcpClientIdentity& origin,
                           const BootpDhcpMessage* pRequest,
                           size_t requestSize) NN_NOEXCEPT;

    /**
     * @brief Event handler for new messages containing a
     * @ref DhcpOptionCode::DhcpMessageType option value of
     * @ref DhcpMessageType::Decline.
     *
     * @param[out] pOutResponse The response object.
     *
     * @param[in] responseSize The size of of the response object in
     * bytes.
     *
     * @param[in] origin The identity of the client that originated the
     * message.
     *
     * @param[in] pRequest The parsed DHCP bootp request message
     * containing option values.
     *
     * @param[in] requestSize The size of the request.
     *
     * @return
     * - rc > 0: Success. The buffer contains rc bytes as a response.
     * - rc == 0: Success. The buffer contains no bytes.
     * - -1 == 0: Failure
     */
    ssize_t OnDhcpDecline(BootpDhcpMessage* pOutResponse,
                          size_t responseSize,
                          const DhcpClientIdentity& origin,
                          const BootpDhcpMessage* pRequest,
                          size_t requestSize) NN_NOEXCEPT;

    /**
     * @brief Event handler for new messages containing a
     * @ref DhcpOptionCode::DhcpMessageType option value of
     * @ref DhcpMessageType::Request.
     *
     * @param[out] pOutResponse The response object.
     *
     * @param[in] responseSize The size of of the response object in
     * bytes.
     *
     * @param[in] origin The identity of the client that originated the
     * message.
     *
     * @param[in] pRequest The parsed DHCP bootp request message
     * containing option values.
     *
     * @param[in] requestSize The size of the request.
     *
     * @return
     * - rc > 0: Success. The buffer contains rc bytes as a response.
     * - rc == 0: Success. The buffer contains no bytes.
     * - -1 == 0: Failure
     */
    ssize_t OnDhcpRequest(BootpDhcpMessage* pOutResponse,
                          size_t responseSize,
                          const DhcpClientIdentity& origin,
                          const BootpDhcpMessage* pRequest,
                          size_t requestSize) NN_NOEXCEPT;

    /**
     * @brief Event handler for new messages containing a
     * @ref DhcpOptionCode::DhcpMessageType option value of
     * @ref DhcpMessageType::Inform.
     *
     * @param[out] pOutResponse The response object.
     *
     * @param[in] responseSize The size of of the response object in
     * bytes.
     *
     * @param[in] origin The identity of the client that originated the
     * message.
     *
     * @param[in] pRequest The parsed DHCP bootp request message
     * containing option values.
     *
     * @param[in] requestSize The size of the request.
     *
     * @return
     * - rc > 0: Success. The buffer contains rc bytes as a response.
     * - rc == 0: Success. The buffer contains no bytes.
     * - -1 == 0: Failure
     */
    ssize_t OnDhcpInform(BootpDhcpMessage* pOutResponse,
                         size_t responseSize,
                         const DhcpClientIdentity& origin,
                         const BootpDhcpMessage* pRequest,
                         size_t requestSize) NN_NOEXCEPT;

    /**
     * @brief Event handler for new messages containing a
     * @ref DhcpOptionCode::DhcpMessageType option value of
     * @ref DhcpMessageType::Release.
     *
     * @param[out] pOutResponse The response object.
     *
     * @param[in] responseSize The size of of the response object in
     * bytes.
     *
     * @param[in] origin The identity of the client that originated the
     * message.
     *
     * @param[in] pRequest The parsed DHCP bootp request message
     * containing option values.
     *
     * @param[in] requestSize The size of the request.
     *
     * @return
     * - rc > 0: Success. The buffer contains rc bytes as a response.
     * - rc == 0: Success. The buffer contains no bytes.
     * - -1 == 0: Failure
     */
    ssize_t OnDhcpRelease(BootpDhcpMessage* pOutResponse,
                          size_t responseSize,
                          const DhcpClientIdentity& origin,
                          const BootpDhcpMessage* pRequest,
                          size_t requestSize) NN_NOEXCEPT;
    /**
     * @brief Prepares and sends a DHCP response.
     *
     * @param[in] pStack An array of @ref NetworkLayersContainer
     * that contain the layers of the packet.
     *
     * @param[in] count The number of layers in @a pStack.
     */
    void SendDhcpResponse(NetworkLayersContainer pStack[],
                          size_t stackCount) NN_NOEXCEPT;

};

/**
 * @brief Create a Client Identifier hash from the given bootp message
 *
 * @param[in] pMessage A pointer to the bootp message.
 *
 * @param[in] size The size of the bootp message.
 *
 * @return The client identifier hash or 0 if the option is not present.
 */
ClientIdentifierHash ClientIdHashFromMessage(const BootpDhcpMessage* pMessage, size_t size) NN_NOEXCEPT;

/**
 * @brief Copies the bytes present in a specific option value into the
 * buffer pointed by pOutValue.
 *
 * @details No byte order conversion is performed.
 *
 * @param[out] pOutValue The out pointer to copy bytes to.
 *
 * @param[in] valueSize The size of the buffer pointed to by @a pOutValue.
 *
 * @param[in] pBuffer A pointer to the DHCP options buffer in the message.
 *
 * @param[in] bufferSize The size of the options buffer.
 *
 * @param[in] code The DhcpOptionCode value corresponding to the
 * desired option.
 *
 * @return On success return 0, on error return -1.
 */
int CopyOptionValue(void* pOutValue,
                    size_t valueSize,
                    const uint8_t* pBuffer,
                    size_t bufferSize,
                    DhcpOptionCode code) NN_NOEXCEPT;

/**
 * @brief This function creates a negative-acknowledgment response.
 *
 * @param[out] pOutResponse The response object.
 *
 * @param[in] responseSize The response object size.
 *
 * @param[in] pRequest The request object.
 *
 * @return On success return 0, on error return -1.
 */
ssize_t CreateNakResponse(BootpDhcpMessage* pOutResponse,
                          size_t responseSize,
                          const BootpDhcpMessage* pRequest) NN_NOEXCEPT;

/**
 * @brief Creates a typical "canned" response containing static
 * data for for the provided message type and lease record.
 *
 * @param[out] pOutResponse The response object.
 *
 * @param[in] responseSize The response object size.
 *
 * @param[in] pRequest The request object.
 *
 * @param[in] messageType The message type of the response.
 *
 * @param[in] pRecord The lease record corresponding to the response.
 * The nullptr value is allowed but lease information will be omitted.
 *
 * @return On success return 0, on error return -1.
 */
ssize_t CreateCannedResponse(BootpDhcpMessage* pOutResponse,
                             size_t responseSize,
                             const BootpDhcpMessage* pRequest,
                             DhcpMessageType messageType,
                             const LeaseRecord* pRecord) NN_NOEXCEPT;

}}}; // nn::dhcps::detail
