﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_Coordinator.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "Coordinator"
#include "dhcps_Log.h"

#include <functional>

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;

namespace {

template <typename T>
int ArrayMax(T& outMaxValue, T pArray[], size_t arraySize) NN_NOEXCEPT
{
    int rc = -1;
    if (pArray == nullptr)
    {
        goto bail;
    };

    outMaxValue = pArray[0];

    if (arraySize > 1)
    {
        for (unsigned int idx=1; idx<arraySize; ++idx)
        {
            outMaxValue = std::max(outMaxValue, pArray[idx]);
        };
    };

    rc = 0;

bail:
    return rc;
};

bool TimeValLess(const nn::socket::TimeVal& a, const nn::socket::TimeVal& b) NN_NOEXCEPT
{
    if (a.tv_sec > b.tv_sec)
    {
        return false;
    }
    else if (a.tv_sec == b.tv_sec)
    {
        return a.tv_usec < b.tv_usec;
    }
    return true;
};

int TimevalArrayMin(nn::socket::TimeVal* pOutMinValue, nn::socket::TimeVal pArray[], size_t arraySize) NN_NOEXCEPT
{
    int rc = -1;
    if (pArray == nullptr || pOutMinValue == nullptr)
    {
        goto bail;
    };

    *pOutMinValue = pArray[0];

    if (arraySize > 1)
    {
        for (unsigned int idx = 1; idx < arraySize; ++idx)
        {
            nn::socket::TimeVal& tv = pArray[idx];
            if (TimeValLess(tv, *pOutMinValue))
            {
                *pOutMinValue = tv;
            };
        };
    };

    rc = 0;

bail:
    return rc;
};

}; // end anonymous namespace

const char* CoordinatorStateToString(Coordinator::State in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::InitializingBpfManager);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::InitializingUdpManager);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::InitializingLeaseTableManager);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::InitializingDhcpManager);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Opening);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::OpenSuccess);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Running);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Stopping);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Stopped);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(Coordinator::State::Error);
    default:
        ;
    };
    return "Unknown State";
};

Coordinator::Coordinator()  NN_NOEXCEPT :
m_State(State::Uninitialized)
{
};

Coordinator::~Coordinator() NN_NOEXCEPT
{
};

int Coordinator::Initialize(LeaseTableManager* pLeaseTableManager) NN_NOEXCEPT
{
    int rc = -1;

    ChangeState(State::InitializingBpfManager);
    if (-1 == (m_BpfManager.Initialize(this)))
    {
        ChangeState(State::Error);
        goto bail;
    };

    ChangeState(State::InitializingUdpManager);
    if ( -1 == (m_UdpSocketManager.Initialize(this)))
    {
        ChangeState(State::Error);
        goto bail;
    };

    m_pLeaseTableManager = pLeaseTableManager;
    ChangeState(State::InitializingLeaseTableManager);
    if ( -1 == (m_pLeaseTableManager->Initialize(this)))
    {
        ChangeState(State::Error);
        goto bail;
    };

    ChangeState(State::InitializingDhcpManager);
    if ( -1 == (m_DhcpManager.Initialize(this, m_pLeaseTableManager)))
    {
        ChangeState(State::Error);
        goto bail;
    };

    ChangeState(State::Initialized);
    rc = 0;

bail:
    return rc;
};

int Coordinator::Finalize() NN_NOEXCEPT
{
    int rc = 0;
    if (-1 == m_BpfManager.Finalize())
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = -1;
    };

    if ( -1 == m_UdpSocketManager.Finalize())
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = -1;
    };

    if (nullptr != m_pLeaseTableManager)
    {
        if (-1 == m_pLeaseTableManager->Finalize())
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("\n");
            rc = -1;
        };
    };

    if ( -1 == m_DhcpManager.Finalize() )
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = -1;
    };

    ChangeState(State::Uninitialized);
    return rc;
};

void Coordinator::OnEvent(const InternalEvent& e) NN_NOEXCEPT
{
    if (EventType::OnTimerExpired != e.GetType())
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("e: {type=%s (%u), value: %p, size: %zu}\n",
                                       EventTypeToString(e.GetType()),
                                       e.GetType(),
                                       e.GetValue(),
                                       e.GetSize());
    };

    switch (e.GetType())
    {
    case EventType::OnPacketRead:
        m_DhcpManager.OnEvent(e);
        break;
    case EventType::OnPacketWrite:
        m_BpfManager.OnEvent(e);
        break;
    case EventType::OnFileError:
    {
        m_BpfManager.OnEvent(e);
        m_DhcpManager.OnEvent(e);
        ChangeState(State::Error);
        break;
    }
    default:
        NN_DETAIL_DHCPS_LOG_DEBUG("Unhandled internal event: %s (%u)\n",
                                  EventTypeToString(e.GetType()),
                                  e.GetType());
        break;
    };

    if (EventTypeIsPublic(e.GetType()))
    {
        if (nullptr != g_Config.GetCallback())
        {
            InternalUnlock();
            g_Config.GetCallback()(g_Config.GetContext(),
                                   static_cast<Event>(static_cast<uint32_t>(e.GetType())),
                                   e.GetValue());
            InternalLock();
        };
    };
};

int Coordinator::Run() NN_NOEXCEPT
{
    InternalLock();
    ApiUnlock();

    ssize_t rc = 0;

    if ( m_State != State::Initialized )
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Did not reach State::Initialized\n");
        ChangeState(State::Error);
        rc = -1;
        goto bail;
    };

    OnFileOpen();

    if ( m_State != State::OpenSuccess )
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Did not reach State::OpenSuccess\n");
        ChangeState(State::Error);
        rc = -1;
        goto bail;
    };

    // check state_ready

    ChangeState(State::Running);

    for (;;)
    {
        if (State::Running != m_State)
        {
            // lock already taken
            break;
        };

        nn::socket::FdSet readfds;
        int max;

        const char* entities[] = {
            "bpf",
            "udp"
        };

        int fdArray[] = {
            m_BpfManager.GetFileDescriptor(),
            m_UdpSocketManager.GetFileDescriptor()
        };

        nn::socket::TimeVal tv = { 0, 0 };

        for (unsigned int idx = 0; idx < sizeof(fdArray) / sizeof(int); ++idx)
        {
            if (-1 == fdArray[idx])
            {
                NN_DETAIL_DHCPS_LOG_MAJOR("Unable to get %s file descriptor\n", entities[idx]);
                // lock taken
                goto bail;
            };
        };

        if ( -1 == ArrayMax(max, fdArray , sizeof(fdArray) / sizeof(int)) )
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("\n");
            // lock taken
            goto bail;
        };

        FD_ZERO(&readfds);
        for (unsigned int idx = 0; idx < sizeof(fdArray) / sizeof(int); ++idx)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("entity: %s, fd: %d\n", entities[idx], fdArray[idx]);
            FD_SET(fdArray[idx], &readfds);
        };

        GetTimeout(&tv);

        InternalUnlock();
        if (-1 == (rc = nn::socket::Select(max + 1,
                                           &readfds,
                                           nullptr,
                                           nullptr,
                                           &tv)))
        {
            NN_DETAIL_DHCPS_LOG_MINOR("Select data error! %s(%d)\n", strerror(errno), errno);
            InternalLock(); // balance lock
            goto bail;
        };
        InternalLock();

        if (0 != rc)
        {
            OnFileRead(&readfds, fdArray[0], fdArray[1]);
        };
        OnTimerExpired();
    };

    rc = 0;

bail:
    InternalUnlock();
    ChangeState(State::Stopped);
    return rc;
};

void Coordinator::InternalStop() NN_NOEXCEPT
{
    ChangeState(State::Stopping);
};

void Coordinator::ExternalStop() NN_NOEXCEPT
{
    InternalLock();
    ChangeState(State::Stopping);
    InternalUnlock();
};

void Coordinator::ChangeState(State next) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("Changing state from: %s (%u) to %s (%u)\n",
                                   CoordinatorStateToString(m_State),
                                   m_State,
                                   CoordinatorStateToString(next),
                                   next);

    SetGlobalState(GlobalState::Coordinator, static_cast<uint8_t>(next));

    m_State = next;

    if (nullptr != g_Config.GetCallback())
    {
        switch (m_State)
        {
        case State::Running:
            g_Config.GetCallback()(g_Config.GetContext(),
                                   Event::OnRunning,
                                   nullptr);
            break;

        case State::Stopping:
            g_Config.GetCallback()(g_Config.GetContext(),
                                   Event::OnStopping,
                                   nullptr);
            break;

        case State::Stopped:
            g_Config.GetCallback()(g_Config.GetContext(),
                                   Event::OnStopped,
                                   nullptr);
            break;

        case State::Error:
            g_Config.GetCallback()(g_Config.GetContext(),
                                   Event::OnUnrecoverableError,
                                   nullptr);
            break;

        default:
            ;
        };
    };

    return;
};

void Coordinator::GetTimeout(nn::socket::TimeVal* outTimeval) NN_NOEXCEPT
{
    const unsigned int MaxEntities = 4;
    nn::socket::TimeVal timeouts[MaxEntities];

    m_BpfManager.GetTimeout(&timeouts[0]);
    m_UdpSocketManager.GetTimeout(&timeouts[1]);
    m_DhcpManager.GetTimeout(&timeouts[2]);
    m_pLeaseTableManager->GetTimeout(&timeouts[3]);

    TimevalArrayMin(outTimeval, timeouts, sizeof(timeouts)  / sizeof(timeouts[0]));
};

void Coordinator::OnFileOpen() NN_NOEXCEPT
{
    ChangeState(State::Opening);

    InternalEvent e(EventType::OnFileOpen, nullptr, 0);

    if (m_State == State::Opening)
    {
        m_BpfManager.OnEvent(e);
    };

    // the state might have changed, check again

    if (m_State == State::Opening)
    {
        m_UdpSocketManager.OnEvent(e);
    };

    ChangeState(State::OpenSuccess);
};

void Coordinator::OnFileRead(nn::socket::FdSet* pReadfds, int bpf, int udp) NN_NOEXCEPT
{
    InternalEvent e(EventType::OnFileRead, nullptr, 0);

    if (State::Running == m_State && nn::socket::FdSetIsSet(bpf, pReadfds))
    {
        m_BpfManager.OnEvent(e);
    };

    // the state might have changed, check again

    if (State::Running == m_State && nn::socket::FdSetIsSet(udp, pReadfds))
    {
        m_UdpSocketManager.OnEvent(e);
    };
};

void Coordinator::OnTimerExpired() NN_NOEXCEPT
{
    InternalEvent e(EventType::OnTimerExpired, nullptr, 0);

    if (State::Running == m_State)
    {
        m_BpfManager.OnEvent(e);
    };

    if (State::Running == m_State)
    {
        m_UdpSocketManager.OnEvent(e);
    };

    if (State::Running == m_State)
    {
        m_DhcpManager.OnEvent(e);
    };

    if (State::Running == m_State)
    {
        m_pLeaseTableManager->OnEvent(e);
    };
};

}}}; // nn::dhcps::detail
