﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dbg/dbg_Api.h>
#include <nn/dbg/dbg_Result.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Result.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/pm/pm_Result.h>
#include <nn/ldr/ldr_DebugMonitorApi.h>
#include <nn/ldr/ldr_Result.h>
#include <nn/ro/ro_DebugMonitorApi.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/util/util_IFunction.h>

namespace nn {
namespace dbg {

    namespace
    {
        Result SetProgramArgument(ncm::ProgramId programId, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
        {
            nn::Result result;

            result = nn::ldr::SetProgramArgument2(programId, pArgument, argumentSize);
            if( result <= nn::ldr::ResultArgumentCountOverflow() )
            {
                result = nn::ldr::FlushArguments2();
                if( result.IsSuccess() )
                {
                    result = nn::ldr::SetProgramArgument2(programId, pArgument, argumentSize);
                }
            }
            return result;
        }

        void FlushProgramArgument(ncm::ProgramId programId) NN_NOEXCEPT
        {
            // 直前の引数を残さないように argv[0] に既定値の空文字列を上書きしておく
            nn::ldr::SetProgramArgument2(programId, "", 1);
        }

        bool Wait(os::SystemEvent* pTargetEvent, os::Event* pCancelEvent)
        {
            if( pCancelEvent != NULL )
            {
                os::MultiWaitType multiWait;
                os::MultiWaitHolderType targetHolder;
                os::MultiWaitHolderType cancelHolder;

                os::InitializeMultiWait(&multiWait);
                os::InitializeMultiWaitHolder(&targetHolder, pTargetEvent->GetBase());
                os::InitializeMultiWaitHolder(&cancelHolder, pCancelEvent->GetBase());
                os::LinkMultiWaitHolder(&multiWait, &targetHolder);
                os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

                bool isSignaled = false;

                for(;;)
                {
                    os::WaitAny(&multiWait);

                    if( pTargetEvent->TryWait() )
                    {
                        isSignaled = true;
                        break;
                    }
                    else if( pCancelEvent->TryWait() )
                    {
                        isSignaled = false;
                        break;
                    }
                }

                os::UnlinkAllMultiWaitHolder(&multiWait);
                os::FinalizeMultiWaitHolder(&cancelHolder);
                os::FinalizeMultiWaitHolder(&targetHolder);
                os::FinalizeMultiWait(&multiWait);

                return isSignaled;
            }
            else
            {
                pTargetEvent->Wait();
                return true;
            }
        }

        Result DebugNext(
            nn::svc::Handle*                                    pOut,
            const util::IFunction<Result(os::NativeHandle*)>&   hook,
            const util::IFunction<Result(os::ProcessId*)>&      getProcessId,
            const util::IFunction<void()>&                      clearHook,
            os::Event*                                          pCancelEvent) NN_NOEXCEPT
        {
            Result result;

            os::NativeHandle eventHandle;
            result = hook(&eventHandle);
            if( result.IsFailure() )
            {
                return result;
            }

            os::SystemEvent hookedEvent;
            hookedEvent.AttachReadableHandle(eventHandle, true, os::EventClearMode_AutoClear);

            os::ProcessId processId;

            for(;;)
            {
                bool isSignaled = Wait(&hookedEvent, pCancelEvent);

                if( isSignaled )
                {
                    // 対象のプログラムが起動した

                    result = getProcessId(&processId);
                    if( result.IsSuccess() )
                    {
                        break;
                    }

                    // スプリアスなシグナルがありうるのでその場合リトライする
                    if( ! (result <= pm::ResultProcessNotFound()) )
                    {
                        break;
                    }

                    hookedEvent.Clear();
                }
                else
                {
                    // キャンセルされた

                    result = ResultCanceled();
                    break;
                }
            }

            clearHook();

            if( result.IsSuccess() )
            {
                svc::Handle debugHandle;

                result = DebugActiveProcess(&debugHandle, processId.value);
                if( result.IsFailure() )
                {
                    return result;
                }

                result = nn::pm::StartProcess(processId);
                if( result.IsFailure() )
                {
                    return result;
                }

                *pOut = debugHandle;
            }

            return result;
        }
    }

    Result Initialize() NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::pm::InitializeForDebugMonitor());
        NN_RESULT_DO(nn::ldr::InitializeForDebugMonitor());
        NN_RESULT_DO(nn::ro::InitializeForDebugMonitor());
        NN_RESULT_DO(nn::ns::InitializeForDevelop());
        NN_RESULT_SUCCESS;
    }

    Result InitializeForSnapShotDumper() NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::ldr::InitializeForDebugMonitor());
        NN_RESULT_DO(nn::ro::InitializeForDebugMonitor());
        NN_RESULT_SUCCESS;
    }

    Result Finalize() NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::ns::FinalizeForDevelop());
        NN_RESULT_DO(nn::ro::FinalizeForDebugMonitor());
        NN_RESULT_DO(nn::ldr::FinalizeForDebugMonitor());
        NN_RESULT_DO(nn::pm::FinalizeForDebugMonitor());
        NN_RESULT_SUCCESS;
    }

    Result FinalizeForSnapShotDumper() NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::ro::FinalizeForDebugMonitor());
        NN_RESULT_DO(nn::ldr::FinalizeForDebugMonitor());
        NN_RESULT_SUCCESS;
    }

    Result DebugNewProcessFromHost(nn::svc::Handle* pOut, const char* pPath, const void* pArgument, size_t size) NN_NOEXCEPT
    {
        nn::Result result;

        ns::ProgramLaunchProperty launchProperty;
        result = ns::PrepareLaunchProgramFromHost( &launchProperty, pPath );
        if( result.IsFailure() )
        {
            return result;
        }

        NN_RESULT_DO(SetProgramArgument(launchProperty.programId, pArgument, size));

        // Start なしでプログラム起動
        nn::os::ProcessId processId;
        int flags = ns::LaunchProgramFlags_NotStart
                  | ns::LaunchProgramFlags_DisableAslr
                  | ns::LaunchProgramFlags_NotifyDebug
                  | ns::LaunchProgramFlags_NotifyStart
                  | ns::LaunchProgramFlags_NotifyExit;
        result = nn::ns::LaunchProgram(&processId, launchProperty, flags);
        FlushProgramArgument(launchProperty.programId);
        if( result.IsFailure() )
        {
            return result;
        }

        // アタッチ
        nn::svc::Handle debugHandle;
        result = DebugActiveProcess(&debugHandle, processId.value);
        if( result.IsFailure() )
        {
            if( result <= svc::ResultInvalidState() )
            {
                ns::TerminateProcess(processId);
                return dbg::ResultCannotDebug();
            }
            return result;
        }

        // Start
        result = nn::pm::StartProcess(processId);
        if( result.IsFailure() )
        {
            return result;
        }

        *pOut = debugHandle;
        return ResultSuccess();
    }

    Result DebugNextProcess(nn::svc::Handle* pOut, ncm::ProgramId programId) NN_NOEXCEPT
    {
        return DebugNextProcess(pOut, programId, NULL);
    }

    Result DebugNextProcess(nn::svc::Handle* pOut, ncm::ProgramId programId, os::Event* pCancelEvent) NN_NOEXCEPT
    {
        return DebugNext(pOut,
            util::MakeIFunction( [programId](os::NativeHandle* pOutHandle)
            {
                return pm::HookToCreateProcess(pOutHandle, programId);
            } ),
            util::MakeIFunction( [programId](os::ProcessId* pOutProcessId)
            {
                return pm::GetProcessId(pOutProcessId, programId);
            } ),
            util::MakeIFunction( []
            {
                pm::ClearHook(pm::HookType_ProgramId);
            } ),
            pCancelEvent );
    }

    Result DebugNextApplicationProcess(nn::svc::Handle* pOut) NN_NOEXCEPT
    {
        return DebugNextApplicationProcess(pOut, NULL);
    }

    Result DebugNextApplicationProcess(nn::svc::Handle* pOut, os::Event* pCancelEvent) NN_NOEXCEPT
    {
        return DebugNext(pOut,
            util::MakeIFunction( [](os::NativeHandle* pOutHandle)
            {
                return pm::HookToCreateApplicationProcess(pOutHandle);
            } ),
            util::MakeIFunction( [](os::ProcessId* pOutProcessId)
            {
                return pm::GetApplicationProcessId(pOutProcessId);
            } ),
            util::MakeIFunction( []
            {
                pm::ClearHook(pm::HookType_Application);
            } ),
            pCancelEvent );
    }

    Result GetProcessModuleInfo(int* pOutCount, dbg::ModuleInfo* pOutModules, int num, os::ProcessId pid) NN_NOEXCEPT
    {
        nn::Result result;
        union
        {
            nn::ldr::ModuleInfo ldrModules[MaxNsoModules];
            nn::ro::ModuleInfo roModules[MaxNroModules];
        } buffer;

        int count = 0;
        int index = 0;
        result = ldr::GetProcessModuleInfo(&count, buffer.ldrModules, NN_ARRAY_SIZE(buffer.ldrModules), pid);
        if (result.IsSuccess())
        {
            for (int i = 0; index < num && i < count; i++)
            {
                nn::ldr::ModuleInfo* pInfo = &buffer.ldrModules[i];
                NN_STATIC_ASSERT(sizeof(pInfo->moduleId) == sizeof(pOutModules[index].moduleId));
                std::memcpy(pOutModules[index].moduleId, pInfo->moduleId, sizeof(pInfo->moduleId));
                pOutModules[index].address = pInfo->address;
                pOutModules[index].size = pInfo->size;
                index++;
            }

            result = ro::GetProcessModuleInfo(&count, buffer.roModules, NN_ARRAY_SIZE(buffer.roModules), pid);
            if (result.IsSuccess())
            {
                for (int i = 0; index < num && i < count; i++)
                {
                    nn::ro::ModuleInfo* pInfo = &buffer.roModules[i];
                    NN_STATIC_ASSERT(sizeof(pInfo->moduleId) == sizeof(pOutModules[index].moduleId));
                    std::memcpy(pOutModules[index].moduleId, pInfo->moduleId, sizeof(pInfo->moduleId));
                    pOutModules[index].address = pInfo->address;
                    pOutModules[index].size = pInfo->size;
                    index++;
                }
            }
        }
        *pOutCount = index;
        return result;
    }

    Result DebugActiveProcess(nn::svc::Handle* pOut, nn::Bit64 processId) NN_NOEXCEPT
    {
        auto result = svc::DebugActiveProcess( pOut, processId );
        if( result <= svc::ResultBusy() )
        {
            return dbg::ResultAlreadyAttached();
        }
        return result;
    }

    Result DebugActiveProcess(nn::svc::Handle* pOut, ncm::ProgramId programId) NN_NOEXCEPT
    {
        nn::Result result;

        nn::os::ProcessId processId;
        result = nn::pm::GetProcessId(&processId, programId);
        if( result.IsFailure() )
        {
            return result;
        }

        nn::svc::Handle debugHandle;
        result = DebugActiveProcess(&debugHandle, processId.value);
        if( result.IsFailure() )
        {
            return result;
        }

        *pOut = debugHandle;
        return ResultSuccess();
    }

    Result DebugActiveApplication(nn::svc::Handle* pOut) NN_NOEXCEPT
    {
        nn::Result result;

        nn::os::ProcessId processId;
        result = nn::pm::GetApplicationProcessId(&processId);
        if( result.IsFailure() )
        {
            return result;
        }

        nn::svc::Handle debugHandle;
        result = DebugActiveProcess(&debugHandle, processId.value);
        if( result.IsFailure() )
        {
            return result;
        }

        *pOut = debugHandle;
        return ResultSuccess();
    }

    Result BreakDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
    {
        return nn::svc::BreakDebugProcess( debug );
    }

    Result TerminateDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
    {
        return nn::svc::TerminateDebugProcess( debug );
    }

    Result GetDebugEvent(nn::svc::DebugEventInfo* pInfo, nn::svc::Handle debug) NN_NOEXCEPT
    {
        return nn::svc::GetDebugEvent( pInfo, debug );
    }

    Result ContinueDebugEvent(nn::svc::Handle debug, nn::Bit32 flags, nn::Bit64 threadIds[], nn::Bit32 size) NN_NOEXCEPT
    {
        return nn::svc::ContinueDebugEvent( debug, flags, threadIds, size );
    }

    Result GetProcessList(int32_t* pNumProcesses, nn::Bit64 pProcessIds[], int32_t arraySize) NN_NOEXCEPT
    {
        return nn::svc::GetProcessList( pNumProcesses, pProcessIds, arraySize );
    }

    Result GetThreadList(int32_t* pNumThreads, nn::Bit64 pThreadIds[], int32_t arraySize, nn::svc::Handle domain) NN_NOEXCEPT
    {
        return nn::svc::GetThreadList( pNumThreads, pThreadIds, arraySize, domain );
    }

    Result GetDebugThreadContext(nn::svc::ThreadContext* pContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags) NN_NOEXCEPT
    {
        return nn::svc::GetDebugThreadContext( pContext, debug, threadId, controlFlags );
    }

    Result SetDebugThreadContext(nn::svc::Handle debug, nn::Bit64 threadId, const nn::svc::ThreadContext& context, nn::Bit32 controlFlags) NN_NOEXCEPT
    {
        return nn::svc::SetDebugThreadContext( debug, threadId, context, controlFlags );
    }

    Result QueryDebugProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr) NN_NOEXCEPT
    {
        return nn::svc::QueryDebugProcessMemory( pBlockInfo, pPageInfo, process, addr );
    }

    Result ReadDebugProcessMemory(uintptr_t buf, nn::svc::Handle debug, uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        return nn::svc::ReadDebugProcessMemory( buf, debug, addr, size );
    }

    Result WriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t buf, uintptr_t addr, size_t size) NN_NOEXCEPT
    {
        return nn::svc::WriteDebugProcessMemory( debug, buf, addr, size );
    }

    Result SetHardwareBreakPoint(nn::svc::HardwareBreakPointRegisterName regNo, nn::Bit64 control, nn::Bit64 value) NN_NOEXCEPT
    {
        return nn::svc::SetHardwareBreakPoint( regNo, control, value );
    }

    Result GetDebugThreadParam(nn::Bit64* pOut1, nn::Bit32* pOut2, nn::svc::Handle debug, nn::Bit64 threadId, nn::svc::DebugThreadParam select) NN_NOEXCEPT
    {
        return nn::svc::GetDebugThreadParam( pOut1, pOut2, debug, threadId, select );
    }

    Result CloseHandle(nn::svc::Handle handle) NN_NOEXCEPT
    {
        return nn::svc::CloseHandle( handle );
    }

    Result WaitSynchronization(int32_t* pOut, const nn::svc::Handle handles[], int32_t numHandles, int64_t ns) NN_NOEXCEPT
    {
        return nn::svc::WaitSynchronization( pOut, handles, numHandles, ns );
    }

    Result GetCurrentProcessId(nn::Bit64* pOut) NN_NOEXCEPT
    {
        return nn::svc::GetProcessId( pOut, static_cast<nn::svc::Handle>(svc::PSEUDO_HANDLE_CURRENT_PROCESS.value) );
    }

    Result GetProcessId(nn::Bit64* pOut, nn::svc::Handle process) NN_NOEXCEPT
    {
        return nn::svc::GetProcessId( pOut, process );
    }

    Result GetJitDebugProcessList(int32_t* pNumProcesses, os::ProcessId pProcessIds[], int32_t arraySize) NN_NOEXCEPT
    {
        return pm::GetExceptionProcessIdList(pNumProcesses, pProcessIds, arraySize);
    }

} // end of namespace dbg
} // end of namespace nn
