﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/i2c/i2c.h>
#include <nn/cpad/cpad.h>

#include "cpad_Util.h"


namespace nn {
namespace cpad {
namespace detail {

/**
 * @brief アナログスティックの入力変化の有無を返す関数
 */
bool HasStickChange( const CpadStickState& pre, const CpadStickState& post, int threshold )
{
    return !( post.x > pre.x - threshold  && post.x < pre.x + threshold &&
              post.y > pre.y - threshold  && post.y < pre.y + threshold );
}

/**
 * @brief ディジタルボタンの入力変化の有無を返す関数
 */
bool HasButtonChange( const nn::util::BitPack32& pre, const nn::util::BitPack32& post )
{
    return pre.storage != post.storage;
}

/**
 * @brief   アナログスティック入力の八角形クランプを行います。
 *          デッドゾーン、最大値、入力定義域等はクラコンの仕様に依存します。
 *          Cafe の PADライブラリに準拠しています。
 */
void ClampStickState( CpadStickState* pOutState)
{
    const int DeadZone   = 15; // 無入力として扱う範囲
    const int SlantedMax = 40; // 斜め方向の最大値

    int signX = 1;
    int signY = 1;

    // 第一象限に座標値を変換。符号はあとで補正するので記憶しておく。
    if(0 > pOutState->x)
    {
        signX = -1;
        pOutState->x = -pOutState->x;
    }
    if(0 > pOutState->y)
    {
        signY = -1;
        pOutState->y = -pOutState->y;
    }

    // デッドゾーンの処理
    (pOutState->x <= DeadZone) ? pOutState->x = 0 : pOutState->x -= DeadZone;
    (pOutState->y <= DeadZone) ? pOutState->y = 0 : pOutState->y -= DeadZone;

    // 無入力時はスケーリング不要なので return
    if(pOutState->x == 0 && pOutState->y == 0)
    {
        return;
    }

    // 第一象限におけるスケーリングパラメータを導出
    int divisor = (pOutState->y <= pOutState->x) ?
        (SlantedMax * pOutState->x + (CpadStickMax - SlantedMax) * pOutState->y) :
        (SlantedMax * pOutState->y + (CpadStickMax - SlantedMax) * pOutState->x);

    // 範囲外であればスケーリング
    if(SlantedMax * CpadStickMax < divisor)
    {
        pOutState->x = SlantedMax * CpadStickMax * pOutState->x / divisor;
        pOutState->y = SlantedMax * CpadStickMax * pOutState->y / divisor;
    }

    // 符号の補正
    pOutState->x *= signX;
    pOutState->y *= signY;
}

/**
 * @brief スティックの中央値を設定する関数
 */
void SetStickOrigin( CpadStickState* pInState, const CpadStickState& state )
{
    pInState->x = state.x;
    pInState->y = state.y;
}


} // detail
} // cpad
} // nn
