﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/i2c/i2c.h>
#include <nn/gpio/gpio.h>
#include <nn/cpad/cpad.h>

#include "cpad_Util.h"
#include "cpad_Cpad.h"

namespace nn {
namespace cpad {
namespace detail {

/**
 * @brief Cpad ライブラリの初期化
 */
void Cpad::InitializeCpad() NN_NOEXCEPT
{
    // 既に初期化済の場合は return
    if( m_IsCpadInitialized )
    {
        return;
    }

    nn::i2c::Initialize();
    nn::i2c::OpenSession( &m_I2cSession, CpadName );
// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::gpio::Initialize();
    nn::gpio::OpenSession( &m_GpioSession, nn::gpio::GpioPadName_DebugControllerDet );
    nn::gpio::SetDirection(&m_GpioSession, nn::gpio::Direction_Input);
    // TODO : Close() と Finalize() をどこかに書く
#endif
    m_IsCpadInitialized = true;
}

/**
 * @brief コントローラ入力生データを取得する関数
 *        デバイスが初期化済でない場合は初期化を行う。
 */
Result Cpad::GetRawCpadState(CpadState* pOutState) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsCpadInitialized );

    nn::Bit8   receiveBuf[8];

// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    if(!m_IsDeviceInitialized &&
       nn::gpio::GetValue(&m_GpioSession) == nn::gpio::GpioValue_Low)
    {
        // クラコンが差さってないときは I2C 通信を行わず、Noack としてエラーを変換して返す
        NN_RESULT_DO(MapCpadResult(nn::i2c::ResultNoAck()));
    }
#endif

    // デバイスの初期化
    NN_RESULT_DO(InitializeDevice());

    // コントローラデータ(生値)を取得
    NN_RESULT_DO(GetCpadRegister(receiveBuf, 0x00, sizeof(receiveBuf)));

    // 生値を DataFormat でパースする。レジスタ値のマッピングのみで、スティック入力には一切の補正(中心座標補正・クランプ)を行わない。
    ParseInput( pOutState, receiveBuf, DataFormat);

    NN_RESULT_SUCCESS;
}

/**
 * @brief コントローラ入力を取得する関数
 */
Result Cpad::GetCpadState(CpadState* pOutState) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsCpadInitialized );

    // コントローラ入力の生データを取得
    NN_RESULT_DO(GetRawCpadState( pOutState ));

    // 中心座標値で補正
    pOutState->stickL.x -= m_StickOriginL.x;
    pOutState->stickL.y -= m_StickOriginL.y;
    pOutState->stickR.x -= m_StickOriginR.x;
    pOutState->stickR.y -= m_StickOriginR.y;

    // 8角形クランプを行う
    ClampStickState( &pOutState->stickL );
    ClampStickState( &pOutState->stickR );

    NN_RESULT_SUCCESS;
}

/**
 * @brief 指定された入力状態列から入力変化を検知する関数
 */
bool   Cpad::HasCpadChange(const CpadState *pInStates, int statesCount) const NN_NOEXCEPT
{
    for(int i = 0; i < statesCount - 1; i++)
    {
        // pInStates[i] と pInStates[i+1] の間で入力変化を検知
        if( HasStickChange ( pInStates[i].stickL,  pInStates[i + 1].stickL, AnalogStickChangeDetectionThreshold ) ||
            HasStickChange ( pInStates[i].stickR,  pInStates[i + 1].stickR, AnalogStickChangeDetectionThreshold ) ||
            HasButtonChange( pInStates[i].buttons, pInStates[i + 1].buttons ) )
        {
            return true;
        }
    }
    return false;
}

/**
 * @brief 指定された状態に入力があるかを検知する関数
 */
bool   Cpad::HasCpadInput(CpadState state) const NN_NOEXCEPT
{
    // 少なくとも一つのボタン・スティック入力が有れば入力有りと判定
    return state.buttons.storage != 0 || state.stickL.x != 0 || state.stickL.y != 0 || state.stickR.x != 0 || state.stickR.y != 0;
}

/**
 * @brief スティックのキャリブレーションを行う関数
 */
bool   Cpad::CalibrateCpad() NN_NOEXCEPT
{
    CpadState rawCpadState;

    auto result = GetRawCpadState( &rawCpadState );

    if( result.IsSuccess() )
    {
        // 中心座標値の設定
        m_StickOriginL.x = rawCpadState.stickL.x;
        m_StickOriginL.y = rawCpadState.stickL.y;
        m_StickOriginR.x = rawCpadState.stickR.x;
        m_StickOriginR.y = rawCpadState.stickR.y;

        return true;
    }

    return false;
}

/**
 * @brief startRegister から outBufSize 分のデータを pOutBuf に格納する。
 */
Result Cpad::GetCpadRegister(Bit8* pOutBuf, nn::Bit8 startRegister, size_t outBufSize) NN_NOEXCEPT
{
    const uint8_t SendBuf             = startRegister;
    uint8_t commandList[nn::i2c::CommandListLengthCountMax];

    nn::i2c::CommandListFormatter  commandListFormatter(commandList, sizeof(commandList));
    std::memset(&commandList, 0, sizeof(commandList));

    commandListFormatter.EnqueueSendCommand( InOption, &SendBuf, 1 );
    commandListFormatter.EnqueueSleepCommand( static_cast<uint8_t>(ReadIntervalMicroSeconds) );
    commandListFormatter.EnqueueReceiveCommand( InOption, outBufSize );

    // コマンドリスト実行
    auto result = ExecuteCommandList( pOutBuf, outBufSize, m_I2cSession, commandList, commandListFormatter.GetCurrentLength() );
    NN_RESULT_DO(MapCpadResult(result));

    NN_RESULT_SUCCESS;
}

/**
 * @brief デバイス(クラシックコントローラ) の初期化を行う関数
 */
Result Cpad::InitializeDevice() NN_NOEXCEPT
{
    nn::Result result;
    nn::Bit8   receiveBuf[2];
    nn::Bit8   setUpData[][2] = {
    //   reg   data
        {0xF0, 0x55},      // Encryption mode を off に設定
        {0xFB, 0x00},      // Vibration を off に設定
        {0xFE, DataFormat} // Data Format を指定
    };

    if( m_IsDeviceInitialized )
    {
        NN_RESULT_SUCCESS; // デバイスが初期化済の場合は成功とする
    }

    for(auto sendData : setUpData )
    {
        result = nn::i2c::Send(m_I2cSession, sendData, sizeof(sendData), InOption );
        NN_RESULT_DO(MapCpadResult(result));
    }

    // Dataformat と Device ID の確認
    // 1回のトランザクションでDataFormat と DeviceId を読み出したいので 2[Bytes] 読み出す
    NN_RESULT_DO(GetCpadRegister(receiveBuf, 0xFE, sizeof(receiveBuf)));

    if( receiveBuf[0] != DataFormat ||
        receiveBuf[1] != DeviceId )
    {
        return nn::cpad::ResultCpadNotFound();
    }

    m_IsDeviceInitialized = true;

    NN_RESULT_SUCCESS;
}

/**
 * @brief 生値入力を指定されたデータフォーマットでパースする関数
 */
void   Cpad::ParseInput( CpadState* pOutState, const Bit8* pInputBuf, int dataFormat) const NN_NOEXCEPT
{
    const int ButtonDataPosition = 6; // ボタンの情報は pInputBuf の 6 バイト目
    nn::util::BitPack8 registerBit;

    switch( dataFormat )
    {
        case DataFormat:
            registerBit.SetMaskedBits(0xFF, pInputBuf[ButtonDataPosition]);

            // 0 ビット目には無効な値が入っているため、考慮しない
            pOutState->buttons.Set<CpadButton::R>      (!registerBit.GetBit(1));
            pOutState->buttons.Set<CpadButton::Start>  (!registerBit.GetBit(2));
            pOutState->buttons.Set<CpadButton::Home>   (!registerBit.GetBit(3));
            pOutState->buttons.Set<CpadButton::Select> (!registerBit.GetBit(4));
            pOutState->buttons.Set<CpadButton::L>      (!registerBit.GetBit(5));
            pOutState->buttons.Set<CpadButton::Down>   (!registerBit.GetBit(6));
            pOutState->buttons.Set<CpadButton::Right>  (!registerBit.GetBit(7));

            registerBit.SetMaskedBits(0xFF, pInputBuf[ButtonDataPosition + 1]);
            pOutState->buttons.Set<CpadButton::Up>     (!registerBit.GetBit(0));
            pOutState->buttons.Set<CpadButton::Left>   (!registerBit.GetBit(1));
            pOutState->buttons.Set<CpadButton::ZR>     (!registerBit.GetBit(2));
            pOutState->buttons.Set<CpadButton::X>      (!registerBit.GetBit(3));
            pOutState->buttons.Set<CpadButton::A>      (!registerBit.GetBit(4));
            pOutState->buttons.Set<CpadButton::Y>      (!registerBit.GetBit(5));
            pOutState->buttons.Set<CpadButton::B>      (!registerBit.GetBit(6));
            pOutState->buttons.Set<CpadButton::ZL>     (!registerBit.GetBit(7));

            pOutState->stickL.x = pInputBuf[0];
            pOutState->stickR.x = pInputBuf[1];
            pOutState->stickL.y = pInputBuf[2];
            pOutState->stickR.y = pInputBuf[3];

            break;
        default:
            NN_ABORT("Invalid data format=%d was set.\n", dataFormat);
            break;
    }
}

/**
 * @brief I2C ドライバライブラリが返すエラーを Cpad エラーに再解釈する関数
 */
Result   Cpad::MapCpadResult(nn::Result i2cError) NN_NOEXCEPT
{
    if(i2cError.IsSuccess())
    {
        return i2cError;
    }

    if( nn::i2c::ResultNoAck::Includes(i2cError) ||
        nn::i2c::ResultBusBusy::Includes(i2cError) )
    {
        // No Ack, BusBusy がきたときは念のためデバイスの初期化をする
        m_IsDeviceInitialized = false;

        return nn::cpad::ResultCpadNotFound();
    }
    else
    {
        // 上記以外はハンドリングのしようがないのでAbort
        NN_ABORT("%s(%d): Unknown error\n", __FUNCTION__ , __LINE__ );
    }
}


} // detail
} // cpad
} // nn
