﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>  // std::memset

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/codec/codec_OpusPacket.h>
#include <nn/codec/codec_OpusMultiStreamDecoder.h>
#include <nn/codec/detail/codec_OpusCommonInternal.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include "opus.h"
#include "opus_multistream.h"

namespace nn {
namespace codec {

namespace {

OpusResult ConvertToOpusResult(int result) NN_NOEXCEPT
{
    switch(result)
    {
    case OPUS_UNIMPLEMENTED:
        return OpusResult_UnsupportedFormat;
    case OPUS_INTERNAL_ERROR:
        return OpusResult_InternalError;
    default:
        return OpusResult_BrokenData;
    }
}

} // anonymous namespace


OpusMultiStreamDecoder::OpusMultiStreamDecoder() NN_NOEXCEPT
    : m_Decoder()
{
}

OpusMultiStreamDecoder::~OpusMultiStreamDecoder() NN_NOEXCEPT
{
}

OpusResult OpusMultiStreamDecoder::Initialize(int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount, const uint8_t channelMapping[], void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeOpusMultiStreamDecoder(&m_Decoder, sampleRate, channelCount, totalStreamCount, stereoStreamCount, channelMapping, buffer, size);
}

void OpusMultiStreamDecoder::Finalize() NN_NOEXCEPT
{
    return FinalizeOpusMultiStreamDecoder(&m_Decoder);
}

size_t GetOpusMultiStreamDecoderWorkBufferSize(int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(totalStreamCount + stereoStreamCount <= channelCount);
    NN_SDK_REQUIRES(totalStreamCount >= 1 && totalStreamCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(stereoStreamCount >= 0 && stereoStreamCount <= totalStreamCount);

    NN_UNUSED(sampleRate);
    NN_UNUSED(channelCount);

    return opus_multistream_decoder_get_size(totalStreamCount, stereoStreamCount);
}

OpusResult InitializeOpusMultiStreamDecoder(OpusMultiStreamDecoderType* pOutDecoder, int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount, const uint8_t channelMapping[], void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDecoder);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(totalStreamCount + stereoStreamCount <= channelCount);
    NN_SDK_REQUIRES(totalStreamCount >= 1 && totalStreamCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(stereoStreamCount >= 0 && stereoStreamCount <= totalStreamCount);
    NN_SDK_REQUIRES(channelMapping != nullptr);
    NN_SDK_REQUIRES(size >= GetOpusMultiStreamDecoderWorkBufferSize(sampleRate, channelCount, totalStreamCount, stereoStreamCount));

    const auto channelMappingMax = static_cast<uint8_t>(totalStreamCount + stereoStreamCount);
    const auto _channelCount = static_cast<uint8_t>(channelCount);
    for (auto k = 0; k < channelCount; ++k)
    {
        if (channelMapping[k] >= channelMappingMax && _channelCount != detail::OpusEmptyChannel)
        {
            return OpusResult_InvalidChannelMapping;
        }
    }
    auto pOpusMsDecoder = static_cast<::OpusMSDecoder*>(buffer);
    if (opus_multistream_decoder_init(pOpusMsDecoder, sampleRate, channelCount, totalStreamCount, stereoStreamCount, channelMapping) != OPUS_OK)
    {
        return OpusResult_OutOfResource;  // TODO: 適切なエラーを返すように
    }

    pOutDecoder->_sampleRate = sampleRate;
    pOutDecoder->_channelCount = channelCount;
    pOutDecoder->_totalStreamCount = totalStreamCount;
    pOutDecoder->_stereoStreamCount = stereoStreamCount;
    pOutDecoder->_buffer = buffer;
    pOutDecoder->_size = size;

    pOutDecoder->_handle = pOpusMsDecoder;

    return OpusResult_Success;
}

void FinalizeOpusMultiStreamDecoder(OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    pDecoder->_handle = nullptr;
    pDecoder->_buffer = nullptr;
    pDecoder->_size = 0;
    pDecoder->_sampleRate = 0;
    pDecoder->_channelCount = 0;
    pDecoder->_totalStreamCount = 0;
    pDecoder->_stereoStreamCount = 0;
}

int GetOpusMultiStreamDecoderSampleRate(const OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_sampleRate;
}

int GetOpusMultiStreamDecoderChannelCount(const OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_channelCount;
}

int GetOpusMultiStreamDecoderTotalStreamCount(const OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_totalStreamCount;
}

int GetOpusMultiStreamDecoderStereoStreamCount(const OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_stereoStreamCount;
}

OpusResult DecodeOpusMultiStreamInterleaved(OpusMultiStreamDecoderType* pDecoder, size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumed);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);

    nn::codec::detail::OpusPacketInternal::Header header;
    const size_t frameHeaderSize = sizeof(header);
    if (inputSize < frameHeaderSize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    auto packet = static_cast<const uint8_t*>(inputBuffer);
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, packet);
    const auto frameBodySize = header.packetSize;
    if (inputSize < frameHeaderSize + frameBodySize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    int sampleCount = 0;
    const auto result = GetOpusPacketSampleCountInPacket(&sampleCount, packet, inputSize, GetOpusMultiStreamDecoderSampleRate(pDecoder));
    if (OpusResult_Success != result)
    {
        return result;
    }
    if (outputSize < sampleCount * GetOpusMultiStreamDecoderChannelCount(pDecoder) * sizeof(int16_t))
    {
        return OpusResult_InsufficientPcmBuffer;
    }
    auto opusFrame = packet + frameHeaderSize;
    // opus_multistream_decoder_ctl と opus_multistream_decode が排他的に呼び出されるように制御する。
    std::lock_guard<nn::os::Mutex> lock(detail::GetOpusGlobalMutex());

    if (pDecoder->_isResetRequested)
    {
        pDecoder->_isResetRequested = false;
        if (const auto ret = opus_multistream_decoder_ctl(static_cast<::OpusMSDecoder*>(pDecoder->_handle), OPUS_RESET_STATE) < 0)
        {
            return ConvertToOpusResult(ret);
        }
    }

    const auto  ret = opus_multistream_decode(
            static_cast<::OpusMSDecoder*>(pDecoder->_handle),
            opusFrame,
            static_cast<int>(frameBodySize),
            outputBuffer,
            static_cast<opus_int32>(outputSize / sizeof(int16_t) / GetOpusMultiStreamDecoderChannelCount(pDecoder)),
            0
    );

    if (ret < 0)
    {
        return ConvertToOpusResult(ret);
    }

    *pOutConsumed = frameHeaderSize + frameBodySize;
    *pOutSampleCount = ret;

    return OpusResult_Success;
}

void ResetOpusMultiStreamDecoder(OpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    pDecoder->_isResetRequested = true;
}

}  // namespace codec
}  // namespace nn
