﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/codec/detail/codec_HardwarePlatform.h>

#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/dd/dd_DeviceAddressSpace.h>

#include <nn/codec/codec_Result.private.h>
#include <nn/codec/detail/codec_IHardwareOpusDecoderManager.h>

#include "codec_CreateHardwareOpusDecoderManager.h"

#else  // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
#include <nn/codec/codec_OpusDecoder.h>
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

#include <nn/codec/codec_OpusPacket.h>
#include <nn/codec/codec_HardwareOpusDecoder.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

namespace nn { namespace codec {

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
NN_STATIC_ASSERT(sizeof(HardwareOpusDecoderType) >= sizeof(OpusDecoderType));
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
namespace {

nn::sf::SharedPointer<detail::IHardwareOpusDecoderManager> CreateHardwareOpusDecoderManager() NN_NOEXCEPT
{
    //return CreateHardwareOpusDecoderManagerByDfc();
    return CreateHardwareOpusDecoderManagerByHipc();
}

class WorkBufferSizeCache
{
public:
    WorkBufferSizeCache() NN_NOEXCEPT
        : m_CachedWorkBufferSizes()
    {
        memset(m_CachedWorkBufferSizes, 0, sizeof(m_CachedWorkBufferSizes));
    }

    size_t GetWorkBufferSize(int sampleRate, int channelCount) const NN_NOEXCEPT
    {
        auto sampleRateIndex = GetSampleRateIndex(sampleRate);
        auto channelIndex = GetChannelIndex(channelCount);
        if (sampleRateIndex == SampleRateIndex_Invalid || channelIndex == ChannelIndex_Invalid)
        {
            return 0;
        }
        return m_CachedWorkBufferSizes[sampleRateIndex][channelIndex];
    }

    void SetWorkBufferSize(int sampleRate, int channelCount, size_t size) NN_NOEXCEPT
    {
        auto sampleRateIndex = GetSampleRateIndex(sampleRate);
        auto channelIndex = GetChannelIndex(channelCount);
        if (sampleRateIndex == SampleRateIndex_Invalid || channelIndex == ChannelIndex_Invalid)
        {
            return;
        }
        NN_SDK_ASSERT(GetWorkBufferSize(sampleRate, channelCount) == 0 || GetWorkBufferSize(sampleRate, channelCount) == size);
        m_CachedWorkBufferSizes[sampleRateIndex][channelIndex] = size;
    }

private:
    enum SampleRateIndex
    {
        SampleRateIndex_Invalid = -1,
        SampleRateIndex_48000 = 0,
        SampleRateIndex_24000 = 1,
        SampleRateIndex_16000 = 2,
        SampleRateIndex_12000 = 3,
        SampleRateIndex_8000  = 4,
        SampleRateIndex_Count,
    };

    enum ChannelIndex
    {
        ChannelIndex_Invalid = -1,
        ChannelIndex_1 = 0,
        ChannelIndex_2 = 1,
        ChannelIndex_Count,
    };

    SampleRateIndex GetSampleRateIndex(int sampleRate) const NN_NOEXCEPT
    {
        switch (sampleRate)
        {
            case 48000:
                return SampleRateIndex_48000;
            case 24000:
                return SampleRateIndex_24000;
            case 16000:
                return SampleRateIndex_16000;
            case 12000:
                return SampleRateIndex_12000;
            case 8000:
                return SampleRateIndex_8000;
            default:
                return SampleRateIndex_Invalid;
        }
    }

    ChannelIndex GetChannelIndex(int channelCount) const NN_NOEXCEPT
    {
        if (channelCount == 1 || channelCount == 2)
        {
            return static_cast<ChannelIndex>(channelCount - 1);
        }
        return ChannelIndex_Invalid;
    }

    size_t m_CachedWorkBufferSizes[SampleRateIndex_Count][ChannelIndex_Count];
};
WorkBufferSizeCache g_WorkBufferSizeCache;

} // anonymous
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

HardwareOpusDecoder::HardwareOpusDecoder(int32_t option) NN_NOEXCEPT
    : m_Decoder()
    , m_Option(option)
{
}

HardwareOpusDecoder::~HardwareOpusDecoder() NN_NOEXCEPT
{
}

OpusResult HardwareOpusDecoder::Initialize(int sampleRate, int channelCount, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeHardwareOpusDecoder(&m_Decoder, sampleRate, channelCount, buffer, size, m_Option);
}

void HardwareOpusDecoder::Finalize() NN_NOEXCEPT
{
    FinalizeHardwareOpusDecoder(&m_Decoder);
}

size_t GetHardwareOpusDecoderWorkBufferSize(int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusDecoderWorkBufferSize(sampleRate, channelCount);
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    auto cached = g_WorkBufferSizeCache.GetWorkBufferSize(sampleRate, channelCount);
    if (cached > 0)
    {
        return cached;
    }

    auto manager = CreateHardwareOpusDecoderManager();
    int size;
    detail::HardwareOpusDecoderParameterInternal parameter =
    {
        sampleRate,
        channelCount,
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager->GetWorkBufferSize(&size, parameter));
    auto ret = static_cast<size_t>(nn::util::align_up(size, nn::os::MemoryPageSize));
    g_WorkBufferSizeCache.SetWorkBufferSize(sampleRate, channelCount, ret);
    return ret;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

OpusResult InitializeHardwareOpusDecoder(HardwareOpusDecoderType* pOutDecoder, int sampleRate, int channelCount, void* buffer, size_t size, int32_t option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDecoder);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= 2);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= GetHardwareOpusDecoderWorkBufferSize(sampleRate, channelCount));
    NN_UNUSED(option);
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    OpusDecoderType* pDecoder = reinterpret_cast<OpusDecoderType*>(pOutDecoder);
    auto result = InitializeOpusDecoder(pDecoder, sampleRate, channelCount, buffer, size);
    if (result != OpusResult_Success)
    {
        return result;
    }
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES(nn::util::is_aligned(reinterpret_cast<uintptr_t>(buffer), nn::os::MemoryPageSize));

    auto manager = CreateHardwareOpusDecoderManager();
    detail::HardwareOpusDecoderParameterInternal parameter =
    {
        sampleRate,
        channelCount,
    };

    nn::os::TransferMemory transferMemory(buffer, size, nn::os::MemoryPermission_None);
    nn::sf::SharedPointer<detail::IHardwareOpusDecoder> decoder;
    auto result = manager->OpenHardwareOpusDecoder(&decoder, parameter, nn::sf::NativeHandle(transferMemory.Detach(), true), size);

    // 適切なステータスを返す必要がある。
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultCpuOutOfResource)
        {
            return OpusResult_OutOfResource;
        }
        // 上記と Success 以外の result は、ここまで返ってこずに abort するはず。
        // NN_RESULT_CATCH_ALL
        // {
        // }
    NN_RESULT_END_TRY

    pOutDecoder->_handle = decoder.Detach();
    pOutDecoder->_buffer = buffer;
    pOutDecoder->_size = size;
    pOutDecoder->_sampleRate = sampleRate;
    pOutDecoder->_channelCount = channelCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

    pOutDecoder->_perf = 0;
    pOutDecoder->_isResetRequested = false;

    return OpusResult_Success;
}

void FinalizeHardwareOpusDecoder(HardwareOpusDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    FinalizeOpusDecoder(reinterpret_cast<OpusDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    nn::sf::ReleaseSharedObject(static_cast<detail::IHardwareOpusDecoder*>(pDecoder->_handle));
    pDecoder->_handle = nullptr;
    pDecoder->_buffer = nullptr;
    pDecoder->_size = 0;
    pDecoder->_sampleRate = 0;
    pDecoder->_channelCount = 0;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusDecoderSampleRate(const HardwareOpusDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusDecoderSampleRate(reinterpret_cast<const OpusDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_sampleRate;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusDecoderChannelCount(const HardwareOpusDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusDecoderChannelCount(reinterpret_cast<const OpusDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_channelCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

OpusResult DecodeOpusInterleavedWithHardware(HardwareOpusDecoderType* pDecoder, size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumed);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return DecodeOpusInterleaved(reinterpret_cast<OpusDecoderType*>(pDecoder), pOutConsumed, pOutSampleCount, outputBuffer, outputSize, inputBuffer, inputSize);
#else // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    nn::codec::detail::OpusPacketInternal::Header header;
    const size_t frameHeaderSize = sizeof(header);  // TODO: ちゃんとした値は後日決めるが、今は opus_demo に合わせておく
    if (inputSize < frameHeaderSize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    auto packet = static_cast<const uint8_t*>(inputBuffer);
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, packet);
    const auto frameBodySize = header.packetSize;
    if (inputSize < frameHeaderSize + frameBodySize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    int sampleCount = 0;
    const auto opusResult = GetOpusPacketSampleCountInPacket(&sampleCount, packet, inputSize, GetHardwareOpusDecoderSampleRate(pDecoder));
    if (OpusResult_Success != opusResult)
    {
        return opusResult;
    }
    if (outputSize < sampleCount * GetHardwareOpusDecoderChannelCount(pDecoder) * sizeof(int16_t))
    {
        return OpusResult_InsufficientPcmBuffer;
    }

    nn::sf::InBuffer inBuffer(static_cast<const char*>(inputBuffer), frameBodySize + frameHeaderSize);
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(outputBuffer), outputSize);

    const auto isResetRequested = pDecoder->_isResetRequested;
    pDecoder->_isResetRequested = false;

    int consumed = 0;
    int outputSampleCount = 0;
    int64_t perf = 0;
    auto result = static_cast<detail::IHardwareOpusDecoder*>(pDecoder->_handle)->DecodeInterleaved(&consumed, &perf, &outputSampleCount, outBuffer, inBuffer, isResetRequested);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultOpusInvalidPacket)
        {
            return OpusResult_BrokenData;
        }
#if 0
        // Result が定義されたら有効にする (TBD)
        NN_RESULT_CATCH(ResultOpusInsufficientPcmBuffer)
        {
            return OpusResult_InsufficientPcmBuffer;
        }
#endif
        // ユーザの管理外なので、ユーザはこのステートを受け取っても何もできない。
        // NN_RESULT_CATCH(ResultOpusInvalidState)
        // {
        //     return OpusResult_InvalidWorkBuffer;
        // }
        // 上記と Success 以外の result は、ここまで返ってこずに abort するはず。
        // NN_RESULT_CATCH_ALL
        // {
        // }
    NN_RESULT_END_TRY

    *pOutConsumed = consumed;
    *pOutSampleCount = outputSampleCount;

    pDecoder->_perf = perf;

    return OpusResult_Success;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

nn::TimeSpan GetHardwareOpusDecoderProcessingTime(const HardwareOpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return nn::TimeSpan::FromMicroSeconds(pDecoder->_perf);
}

void ResetHardwareOpusDecoder(HardwareOpusDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    ResetOpusDecoder(reinterpret_cast<OpusDecoderType*>(pDecoder));
#else
    pDecoder->_isResetRequested = true;
#endif
}

}}  // nn::codec
