﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>  // std::lock_guard

#include <nn/nn_SdkAssert.h>
#include <nn/lmem/lmem_FrameHeap.h>
#include <nn/os/os_Mutex.h>

#include <genericStds.h>  // MEMORY_SECTION

// #define NN_CODEC_AAC_MEMORY_LOG_ENABLED

#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
#include <nn/nn_SdkLog.h>
#endif  // defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)

namespace {

nn::lmem::HeapHandle g_Handle = nullptr;
nn::os::Mutex g_Mutex(false);

#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
int g_Total = 0;
#endif  // defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)

}

namespace nn { namespace codec {

void InitializeAacMemoryAllocator(void* buffer, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    if (g_Handle == nullptr && buffer != nullptr)
    {
        g_Handle = nn::lmem::CreateFrameHeap(buffer, size, nn::lmem::CreationOption_NoOption);
        NN_SDK_ASSERT_NOT_NULL(g_Handle);
#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
        g_Total = 0;
#endif  // defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
    }
}

void FinalizeAacMemoryAllocator() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    if (g_Handle != nullptr)
    {
        nn::lmem::DestroyFrameHeap(g_Handle);
        g_Handle = nullptr;
    }
}

}}  // namespace nn::codec

#if defined(FDK_AAC_NAMESPACE_BEGIN)
FDK_AAC_NAMESPACE_BEGIN
#endif  // defined(FDK_AAC_NAMESPACE_BEGIN)

extern "C" {

void* FDKcalloc(const unsigned int n, const unsigned int size)
{
#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
    g_Total += n * size;
    NN_SDK_LOG("%s (%04d): total = %d, n = %d, size = %d\n", __func__, __LINE__, g_Total, n, size);
#endif
    NN_SDK_ASSERT_NOT_NULL(g_Handle);

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    auto ret = nn::lmem::AllocateFromFrameHeap(g_Handle, n * size);
    std::memset(ret, 0, n * size);
    return ret;
}

void* FDKmalloc(const unsigned int size)
{
#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
    g_Total += size;
    NN_SDK_LOG("%s (%04d): total = %d, size = %d\n", __func__, __LINE__, g_Total, size);
#endif
    NN_SDK_ASSERT_NOT_NULL(g_Handle);

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);
    return nn::lmem::AllocateFromFrameHeap(g_Handle, size);
}

void FDKfree(void* ptr)
{
    NN_UNUSED(ptr);
}

void *FDKaalloc(const unsigned int size, const unsigned int alignment)
{
#if defined(NN_CODEC_AAC_MEMORY_LOG_ENABLED)
    g_Total += size;
    NN_SDK_LOG("%s (%04d): total = %d, size = %d, alignment = %d\n", __func__, __LINE__, g_Total, size, alignment);
#endif
    NN_SDK_ASSERT_NOT_NULL(g_Handle);

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);
    auto ret = nn::lmem::AllocateFromFrameHeap(g_Handle, size, alignment);
    std::memset(ret, 0, size);
    return ret;
}

void FDKafree(void *ptr)
{
    return FDKfree(ptr);
}

void *FDKcalloc_L(const unsigned int dim, const unsigned int size, MEMORY_SECTION s)
{
    if (s == SECT_DATA_EXTERN)
    {
        return FDKcalloc(dim, size);
    }
    else
    {
        return FDKcalloc(dim, nn::util::align_up(size, 4));
    }
}

void FDKfree_L(void *p)
{
    return FDKfree(p);
}

void *FDKaalloc_L(const unsigned int size, const unsigned int alignment, MEMORY_SECTION s)
{
    if (s == SECT_DATA_EXTERN)
    {
        return FDKaalloc(size, alignment);
    }
    else
    {
        return FDKaalloc(nn::util::align_up(size, 4), alignment);
    }
}

void FDKafree_L(void *ptr)
{
    return FDKfree_L(ptr);
}

}  // extern "C"

#if defined(FDK_AAC_NAMESPACE_END)
FDK_AAC_NAMESPACE_END
#endif  // defined(FDK_AAC_NAMESPACE_END)
