﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/codec/codec_AacCommonTypes.h>
#include <nn/codec/codec_AacDecoder.h>
#include <nn/codec/codec_AacDecoderTypes.h>
#include <nn/util/util_ScopeExit.h>

#include "aacdecoder_lib.h"

#include "codec_AacDecoderErrorConverter.h"
#include "codec_AacMemory.h"

#include <mutex>  // std::lock_guard
#include <nn/os/os_Mutex.h>

namespace nn { namespace codec {

namespace {

int GetAudioSpecificConfigSamplingFrequencyIndex(int sampleRate) NN_NOEXCEPT
{
    switch (sampleRate)
    {
        case 96000:
            return 0;
        case 88200:
            return 1;
        case 64000:
            return 2;
        case 48000:
            return 3;
        case 44100:
            return 4;
        case 32000:
            return 5;
        case 24000:
            return 6;
        case 22050:
            return 7;
        case 16000:
            return 8;
        case 12000:
            return 9;
        case 11025:
            return 10;
        case 8000:
            return 11;
        case 7350:
            return 12;
        default:
            return 15;
    }
}

bool IsSupportedSampleRate(int sampleRate) NN_NOEXCEPT
{
    return sampleRate == 96000
        || sampleRate == 88200
        || sampleRate == 64000
        || sampleRate == 48000
        || sampleRate == 44100
        || sampleRate == 32000
        || sampleRate == 24000
        || sampleRate == 22050
        || sampleRate == 16000
        || sampleRate == 12000
        || sampleRate == 11025
        || sampleRate == 8000
        || sampleRate == 7350;
}

bool IsSupportedChannelCount(int channelCount) NN_NOEXCEPT
{
    return (channelCount >= 1 && channelCount <= 6) || channelCount == 8;
}

bool IsSupportedMpeg4AudioObjectType(Mpeg4AudioObjectType type) NN_NOEXCEPT
{
    return type == Mpeg4AudioObjectType_AacLc;
}

nn::os::Mutex g_Mutex(false);

}

AacDecoder::AacDecoder() NN_NOEXCEPT
    : m_Decoder()
{
}

AacDecoder::~AacDecoder() NN_NOEXCEPT
{
}

AacDecoderResult AacDecoder::Initialize(int sampleRate, int channelCount, Mpeg4AudioObjectType type, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeAacDecoder(&m_Decoder, sampleRate, channelCount, type, buffer, size);
}

size_t GetAacDecoderWorkBufferSize(int sampleRate, int channelCount, Mpeg4AudioObjectType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsSupportedSampleRate(sampleRate));
    NN_SDK_REQUIRES(IsSupportedChannelCount(channelCount));
    NN_SDK_REQUIRES(IsSupportedMpeg4AudioObjectType(type));
    NN_UNUSED(sampleRate);
    NN_UNUSED(type);
    size_t size = 0;
    size += 168 * 1024;                // base
    size += 13 * 1024 * channelCount;  // per-channel
    size += 256;                       // management area for lmem::FrameHeap
    return size;
}

AacDecoderResult InitializeAacDecoder(AacDecoderType* pOutDecoder, int sampleRate, int channelCount, Mpeg4AudioObjectType type, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDecoder);
    NN_SDK_REQUIRES(IsSupportedSampleRate(sampleRate));
    NN_SDK_REQUIRES(IsSupportedChannelCount(channelCount));
    NN_SDK_REQUIRES(IsSupportedMpeg4AudioObjectType(type));
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= GetAacDecoderWorkBufferSize(sampleRate, channelCount, type));

    if (!IsSupportedSampleRate(sampleRate))
    {
        return AacDecoderResult_UnsupportedSampleRate;
    }
    if (!IsSupportedChannelCount(channelCount))
    {
        return AacDecoderResult_UnsupportedChannelConfig;
    }
    if (!IsSupportedMpeg4AudioObjectType(type))
    {
        return AacDecoderResult_UnsupportedAudioObjectType;
    }

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    InitializeAacMemoryAllocator(buffer, size);
    NN_UTIL_SCOPE_EXIT
    {
        FinalizeAacMemoryAllocator();
    };

    auto handle = aacDecoder_Open(TT_MP4_ADIF, 1);
    uint16_t tmp = 0;
    tmp |= (static_cast<uint16_t>(type) & 0x1f) << 11;
    tmp |= (static_cast<uint16_t>(GetAudioSpecificConfigSamplingFrequencyIndex(sampleRate)) & 0xf) << 7;
    tmp |= (static_cast<uint16_t>(channelCount == 8 ? 7 : channelCount) & 0xf) << 3;
    uint8_t config[2] = {
        static_cast<uint8_t>((tmp >> 8)),
        static_cast<uint8_t>((tmp >> 0)),
    };

    auto pConfig = config;
    uint32_t configSize = sizeof(config);
    auto result = aacDecoder_ConfigRaw(handle, &pConfig, &configSize);
    if (result != AAC_DEC_OK)
    {
        NN_SDK_ASSERT(result == AAC_DEC_UNSUPPORTED_FORMAT);
        return AacDecoderResult_UnsupportedFormat;
    }

    pOutDecoder->_buffer = buffer;
    pOutDecoder->_size = size;
    pOutDecoder->_sampleRate = sampleRate;
    pOutDecoder->_channelCount = channelCount;
    pOutDecoder->_type = type;
    pOutDecoder->_handle = handle;
    return AacDecoderResult_Success;
}

void FinalizeAacDecoder(AacDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    aacDecoder_Close(reinterpret_cast<HANDLE_AACDECODER>(pDecoder->_handle));
    pDecoder->_handle = nullptr;
    pDecoder->_buffer = nullptr;
    pDecoder->_size = 0;
    pDecoder->_sampleRate = 0;
    pDecoder->_channelCount = 0;
    pDecoder->_type = Mpeg4AudioObjectType_Invalid;
}

int GetAacDecoderSampleRate(const AacDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    return pDecoder->_sampleRate;
}

int GetAacDecoderChannelCount(const AacDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    return pDecoder->_channelCount;
}

Mpeg4AudioObjectType GetAacDecoderMpeg4AudioObjectType(const AacDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

    return pDecoder->_type;
}

AacDecoderResult DecodeAacInterleaved(AacDecoderType* pDecoder, size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumed);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);

    auto decoder = reinterpret_cast<HANDLE_AACDECODER>(pDecoder->_handle);

    auto input = const_cast<uint8_t*>(reinterpret_cast<const uint8_t*>(inputBuffer));
    auto size = static_cast<uint32_t>(inputSize);

    auto result = aacDecoder_Fill(decoder, &input, &size, &size);
    if (result != AAC_DEC_OK)
    {
        NN_SDK_ASSERT(result == AAC_DEC_UNKNOWN);
        aacDecoder_SetParam(decoder, AAC_TPDEC_CLEAR_BUFFER, 0);
        return AacDecoderResult_InternalError;
    }

    uint32_t before;
    aacDecoder_GetFreeBytes(decoder, &before);
    result = aacDecoder_DecodeFrame(decoder, outputBuffer, static_cast<uint32_t>(outputSize), 0);
    if (result != AAC_DEC_OK)
    {
        aacDecoder_SetParam(decoder, AAC_TPDEC_CLEAR_BUFFER, 0);
        return ConvertToAacDecoderResult(result);
    }
    uint32_t after;
    aacDecoder_GetFreeBytes(decoder, &after);
    aacDecoder_SetParam(decoder, AAC_TPDEC_CLEAR_BUFFER, 0);

    if (pOutConsumed)
    {
        *pOutConsumed = after - before;
    }
    if (pOutSampleCount)
    {
        *pOutSampleCount = aacDecoder_GetStreamInfo(decoder)->aacSamplesPerFrame;
    }
    return AacDecoderResult_Success;
}

}}  // namespace nn::codec
