﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/cec/cec_Server.h>

#include <cec_Manager.h>
#include <cec_ManagerImpl.h>
#include <cec_ManagerImpl.private.h>
#include <cec_Error.h>
#include <cec_TopLevelFiniteStateMachine.h>
#include <cec_Receiver.h>
#include <cec_LowLevelInterface.h>
#include "cec_ServerLog.h"

#include "../../hdcp/server/hdcp_HdcpControllerImpl.h"

namespace nn { namespace cec { namespace server {

namespace {

struct EventData
{
    BusEventType    busEvent;
    uint32_t        count;
    uint64_t        eventData0;
    uint64_t        eventData1;
};

struct SystemEventInstance
{
    nn::os::SystemEventType systemEvent;
    uint32_t                handleValue;
};

struct SystemEventState
{
    struct SystemEventInstance  systemEventInstanceArray[maxSessionCount];
    uint32_t                    currentHandleValue;
    uint32_t                    numberOfActiveInstances;
};

const uint32_t InvalidSystemEventHandleValue = 0;
const uint32_t InitialSystemEventHandleValue = 1;

class TopLevelEventHandler
{
    public:
        TopLevelEventHandler() NN_NOEXCEPT
        {
            uint32_t    index;

            m_EventData.busEvent = BusEventType_Ignore;
            m_EventData.count = 0;
            m_EventData.eventData0 = 0;
            m_EventData.eventData1 = 0;
            m_SystemEventState.currentHandleValue = InitialSystemEventHandleValue;
            m_SystemEventState.numberOfActiveInstances = 0;
            for(index=0; index<maxSessionCount; index++)
            {
                m_SystemEventState.systemEventInstanceArray[index].handleValue = InvalidSystemEventHandleValue;
            }
            nn::os::InitializeMutex(&m_SystemEventMutex, false, 0);
        }
        ~TopLevelEventHandler() NN_NOEXCEPT
        {
            nn::os::FinalizeMutex(&m_SystemEventMutex);
        }
        uint32_t AddSystemEventToList(nn::os::SystemEventType** ppSystemEvent) NN_NOEXCEPT
        {
            uint32_t    index;
            uint32_t    rval = InvalidSystemEventHandleValue;

            nn::os::LockMutex(&m_SystemEventMutex);
            *ppSystemEvent = nullptr;
            for(index=0; index < maxSessionCount; index++)
            {
                if(m_SystemEventState.systemEventInstanceArray[index].handleValue == InvalidSystemEventHandleValue)
                {
                    rval = m_SystemEventState.currentHandleValue;
                    m_SystemEventState.currentHandleValue++;
                    m_SystemEventState.numberOfActiveInstances++;
                    nn::os::CreateSystemEvent(&m_SystemEventState.systemEventInstanceArray[index].systemEvent,
                                              nn::os::EventClearMode_AutoClear, true);
                    m_SystemEventState.systemEventInstanceArray[index].handleValue = rval;
                    *ppSystemEvent = &m_SystemEventState.systemEventInstanceArray[index].systemEvent;
                    break;
                }
            }
            nn::os::UnlockMutex(&m_SystemEventMutex);
            return rval;

        }
        int32_t DeleteSystemEventFromList(uint32_t handle) NN_NOEXCEPT
        {
            uint32_t    index;
            int32_t     rval = CecManagerImplOperationFailed;

            nn::os::LockMutex(&m_SystemEventMutex);
            for(index=0; index < maxSessionCount; index++)
            {
                if(m_SystemEventState.systemEventInstanceArray[index].handleValue == handle)
                {
                    nn::os::DestroySystemEvent(&m_SystemEventState.systemEventInstanceArray[index].systemEvent);
                    m_SystemEventState.systemEventInstanceArray[index].handleValue = InvalidSystemEventHandleValue;
                    m_SystemEventState.numberOfActiveInstances--;
                    rval = CecNoError;
                    break;
                }
            }
            nn::os::UnlockMutex(&m_SystemEventMutex);
            return rval;
        }
        int32_t SignalEventInList(uint32_t handle) NN_NOEXCEPT
        {
            uint32_t    index;
            int32_t     rval = CecManagerImplOperationFailed;

            nn::os::LockMutex(&m_SystemEventMutex);
            for(index=0; index < maxSessionCount; index++)
            {
                if(m_SystemEventState.systemEventInstanceArray[index].handleValue == handle)
                {
                    nn::os::SignalSystemEvent(&m_SystemEventState.systemEventInstanceArray[index].systemEvent);
                    rval = CecNoError;
                    break;
                }
            }
            nn::os::UnlockMutex(&m_SystemEventMutex);
            return rval;
        }
        void CecEvent(uint32_t eventFlags, uint32_t count, uint8_t dataBuffer[]) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_SystemEventMutex);
            m_EventData.eventData0 = 0;
            m_EventData.eventData1 = 0;
            m_EventData.count = 0;
            switch(eventFlags)
            {
                case CecEventActiveSource:
                    if(count == 3)
                    {
                        m_EventData.busEvent = (dataBuffer[1] == false) ? BusEventType_ActiveSourceChangedToInactive :
                                                                          BusEventType_ActiveSourceChangedToActive;
                    }
                    else
                    {
                        m_EventData.busEvent = BusEventType_Ignore;
                    }
                    break;
                case CecEventGoStandby:
                    m_EventData.busEvent = BusEventType_GoStandby;
                    break;
                case CecEventHpdConnect:
                    m_EventData.eventData0 = 1;
                case CecEventHpdDisconnect:
                    m_EventData.busEvent = BusEventType_ConnectionChange;
                    m_EventData.count = 1;
                    break;
                case CecEventSuspending:
                    m_EventData.busEvent = BusEventType_Suspending;
                    break;
                case CecEventOtpFeatureAbort:
                    m_EventData.busEvent = BusEventType_FeatureAbortOneTouchPlay;
                    break;
                case CecEventStandbyFeatureAbort:
                    m_EventData.busEvent = BusEventType_FeatureAbortStandby;
                    break;
                case CecEventSetOsdStringFeatureAbort:
                    m_EventData.busEvent = BusEventType_FeatureAbortSetOnScreenString;
                    break;
                case CecEventDeviceMenuControlFeatureAbort:
                    m_EventData.busEvent = BusEventType_FeatureAbortDeviceMenuControl;
                    m_EventData.count = 1;
                    break;
                case CecEventStarted:
                    m_EventData.busEvent = BusEventType_Started;
                    break;
                default:
                    m_EventData.busEvent = BusEventType_Ignore;
                    break;
            }
            if(m_EventData.busEvent != BusEventType_Ignore)
            {
                SignalAllEvents();
            }
            nn::os::UnlockMutex(&m_SystemEventMutex);
        }
        void CopyEventData(struct EventData* pEventData) NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_SystemEventMutex);
            *pEventData = m_EventData;
            nn::os::UnlockMutex(&m_SystemEventMutex);
        }
    private:
        void SignalAllEvents() NN_NOEXCEPT
        {
            uint32_t    index;

            for(index=0; index < maxSessionCount; index++)
            {
                if(m_SystemEventState.systemEventInstanceArray[index].handleValue != InvalidSystemEventHandleValue)
                {
                    nn::os::SignalSystemEvent(&m_SystemEventState.systemEventInstanceArray[index].systemEvent);
                }
            }
        }
        struct EventData        m_EventData;
        struct SystemEventState m_SystemEventState;
        nn::os::MutexType       m_SystemEventMutex;
} s_TopLevelEventHandler;

int32_t CecManagerImplTestPrerequisites() NN_NOEXCEPT
{
    int32_t rval = CecNoError;
    bool    isStarted = false;

    detail::CecTopLevelIsStarted(&isStarted);
    if(isStarted == false)
    {
        rval = CecManagerImplSubsystemSuspended;
    }
    return rval;
}

}

void CecManagerImplOnEvent(uint32_t eventFlags, uint32_t count, uint8_t dataBuffer[]) NN_NOEXCEPT
{
    s_TopLevelEventHandler.CecEvent(eventFlags, count, dataBuffer);
}

int32_t CecManagerImpl::CancelCurrentCall(nn::sf::Out<bool> pCallWasCanceled) NN_NOEXCEPT
{
    bool    wasCanceled = false;

    wasCanceled = detail::CecTopLevelSetCancelState();
    pCallWasCanceled.Set(wasCanceled);
    return CecNoError;
}

nn::Result CecManagerImpl::GetHdcpServiceObject(nn::sf::Out<nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController>> outService) NN_NOEXCEPT
{
    auto p = nn::sf::CreateSharedObjectEmplaced<nn::hdcp::detail::IHdcpController, nn::hdcp::server::HdcpControllerImpl>();
    if (p == nullptr)
    {
        NN_ABORT("Creating HDCP service object failed: service object construction failed.\n");
    }
    outService.Set(p);
    NN_RESULT_SUCCESS;
}

CecManagerImpl::CecManagerImpl() NN_NOEXCEPT
{
    // Hard coded assumptions used elsewhere in the code asserted here
    NN_SDK_ASSERT((detail::CecPowerStatePowerOn == static_cast<uint8_t>(PowerState_On)) &&
                  (detail::CecPowerStateStandby == static_cast<uint8_t>(PowerState_Standby)) &&
                  (detail::CecPowerStatePowerToOn == static_cast<uint8_t>(PowerState_GoingOn)) &&
                  (detail::CecPowerStateToStandby == static_cast<uint8_t>(PowerState_GoingStandby)));
}

CecManagerImpl::~CecManagerImpl() NN_NOEXCEPT
{
}

int32_t CecManagerImpl::RegisterCallback(nn::sf::Out<nn::sf::NativeHandle> registrationEventHandle,
                                         nn::sf::Out<uint32_t> handleValue) NN_NOEXCEPT
{
    nn::os::SystemEventType*    pSystemEvent=nullptr;
    uint32_t                    handle;
    int32_t                     rval = CecManagerImplOperationFailed;

    handle = s_TopLevelEventHandler.AddSystemEventToList(&pSystemEvent);
    if((handle != InvalidSystemEventHandleValue) && (pSystemEvent != nullptr))
    {
        registrationEventHandle.Set(nn::sf::NativeHandle(nn::os::GetReadableHandleOfSystemEvent(pSystemEvent), false));
        nn::os::ClearSystemEvent(pSystemEvent);
        handleValue.Set(handle);
        rval = CecNoError;
    }
    NN_CEC_INFO("%s: handle %d pSysEvent %X --> %X\n", NN_CURRENT_FUNCTION_NAME, handle, pSystemEvent, rval);
    return rval;
}

int32_t CecManagerImpl::UnregisterCallback(uint32_t handleValue) NN_NOEXCEPT
{
    int32_t rval;

    rval = s_TopLevelEventHandler.DeleteSystemEventFromList(handleValue);
    NN_CEC_INFO("%s: %d --> %X\n", NN_CURRENT_FUNCTION_NAME, handleValue, rval);
    return rval;
}

nn::Result CecManagerImpl::TriggerSystemEvent(uint32_t handleValue) NN_NOEXCEPT
{
    s_TopLevelEventHandler.CecEvent(0, 0, nullptr);
    s_TopLevelEventHandler.SignalEventInList(handleValue);
    NN_CEC_INFO("%s: %d\n", NN_CURRENT_FUNCTION_NAME, handleValue);
    NN_RESULT_SUCCESS;
}

int32_t CecManagerImpl::PerformAction(uint32_t action, uint64_t dataZero, uint64_t dataOne) NN_NOEXCEPT
{
    ShimDataTransferType    shimData;
    ActionType              actionType;
    int32_t                 rval;

    NN_CEC_INFO("Enter %s(%d)\n", NN_CURRENT_FUNCTION_NAME, action);
    shimData.ullArray[0] = dataZero;
    shimData.ullArray[1] = dataOne;
    actionType = static_cast<ActionType>(action);
    rval = CecNoError;
    if(detail::CecManagerInit(false))
    {
        switch(actionType)
        {
            case ActionType_OneTouchPlay:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelPerformActiveSource(true);
                    if((rval != CecNoError) && (rval != CecManagerImplCancelledError))
                    {
                        rval = CecManagerImplCommandExecutionFailed;
                    }
                }
                break;
            case ActionType_ImageViewOn:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecReceiverSendImageViewOnCommand();
                }
                break;
            case ActionType_ActiveSource:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecReceiverSendActiveSourceCommand(false);
                }
                break;
            case ActionType_SetActiveSourceState:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    detail::CecTopLevelSetActiveSourceState(true);
                }
                break;
            case ActionType_GoStandby:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelGoStandby(shimData.booleanValue);
                    if((rval != CecNoError) && (rval != CecManagerImplCancelledError))
                    {
                        rval = CecManagerImplCommandExecutionFailed;
                    }
                }
                break;
            case ActionType_InactiveSource:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    detail::CecTopLevelSetActiveSourceState(false);
                    rval = detail::CecTopLevelPerformActiveSource(false);
                    if((rval != CecNoError) && (rval != CecManagerImplCancelledError))
                    {
                        rval = CecManagerImplCommandExecutionFailed;
                    }
                }
                break;
            case ActionType_SetPowerState:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    detail::CecTopLevelUpdatePowerStatus(static_cast<uint8_t>(shimData.powerStateValue));
                }
                break;
            case ActionType_SuspendManager:
                detail::CecTopLevelStop();
                break;
            case ActionType_RestartManager:
                rval = detail::CecTopLevelStart();
                if((rval != CecNoError) && (rval != CecManagerImplCancelledError))
                {
                    uint32_t    hpdStatus;
                    uint32_t    sinkStatus;

                    rval = detail::CecLowLevelInterfaceGetConnectionStatus(&hpdStatus, &sinkStatus);
                    if((rval == CecNoError) && (sinkStatus == 0))
                    {
                        rval = CecManagerImplSubsystemSuspended;
                    }
                    else
                    {
                        rval = CecManagerImplCommandExecutionFailed;
                    }
                }
                break;
            case ActionType_SetOnScreenDisplay:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelSetOnScreenDisplay(shimData.byteArray);
                }
                break;
            case ActionType_RemoteControlCommand:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelSendRemoteControlCommand(static_cast<bool>(shimData.byteArray[0]), shimData.byteArray[1]);
                }
                break;
            case ActionType_DeviceMenuCommand:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelSendDeviceMenuStateCommand(shimData.booleanValue);
                }
                break;
            default:
                rval = CecManagerImplInvalidOperand;
                return rval;
        }
    }
    NN_CEC_INFO("Leave %s -> %X\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
} //NOLINT(impl/function_size)

int32_t CecManagerImpl::QueryState(uint32_t query,
                                   nn::sf::Out<uint64_t> pDataZero,
                                   nn::sf::Out<uint64_t> pDataOne) NN_NOEXCEPT
{
    QueryType   queryType;
    int32_t     rval;
    ShimDataTransferType    shimData;

    NN_CEC_INFO("Enter %s(%d)\n", NN_CURRENT_FUNCTION_NAME, query);
    queryType = static_cast<QueryType>(query);
    rval = CecNoError;
    if(detail::CecManagerInit(false))
    {
        switch(queryType)
        {
            case QueryType_IfStarted:
                rval = detail::CecTopLevelIsStarted(&shimData.booleanValue);
                if(rval != CecNoError)
                {
                    rval = CecManagerImplCommandExecutionFailed;
                }
                break;
            case QueryType_PowerState:
                rval = detail::CecTopLevelGetPowerStatus();
                shimData.powerStateValue = static_cast<PowerState>(rval);
                rval = CecNoError;
                break;
            case QueryType_TvPowerState:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    uint8_t tvPowerState;

                    rval = detail::CecTopLevelGetTvPowerStatus(&tvPowerState);
                    if(rval == CecNoError)
                    {
                        shimData.powerStateValue = static_cast<PowerState>(tvPowerState);
                    }
                }
                break;
            case QueryType_ConnectionState:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    uint32_t    hpdStatus;
                    uint32_t    sinkStatus;

                    rval = detail::CecLowLevelInterfaceGetConnectionStatus(&hpdStatus, &sinkStatus);
                    if(rval == CecNoError)
                    {
                        switch(sinkStatus)
                        {
                            case 0:
                                shimData.connectionStateValue = ConnectionState_NotConnected;
                                break;
                            case 1:
                            case 2:
                                shimData.connectionStateValue = ConnectionState_CradleAndTvConnected;
                                break;
                            default:
                                rval = CecManagerImplInvalidOperand;
                                break;
                        }
                    }
                    else
                    {
                        rval = CecManagerImplCommandExecutionFailed;
                    }
                }
                break;
            case QueryType_HpdState:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    bool    hpdState;

                    rval = detail::CecLowLevelGetHpdState(&hpdState);
                    if(rval == CecNoError)
                    {
                        shimData.booleanValue = hpdState;
                    }
                }
                break;
            case QueryType_IfEnabled:
                shimData.booleanValue = true;
                rval = CecNoError;
                break;
            case QueryType_IfActiveSource:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    shimData.booleanValue = detail::CecTopLevelIsActiveSource();
                }
                break;
            case QueryType_CanPingTv:
                rval = CecManagerImplTestPrerequisites();
                if(rval == CecNoError)
                {
                    rval = detail::CecTopLevelPingTv(&shimData.booleanValue);
                }
                break;
            default:
                rval = CecManagerImplInvalidOperand;
                return rval;
        }
        pDataZero.Set(shimData.ullArray[0]);
        pDataOne.Set(shimData.ullArray[1]);
    }
    else
    {
        pDataZero.Set(0);
        pDataOne.Set(0);
    }
    NN_CEC_INFO("Leave %s -> %X\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
} //NOLINT(impl/function_size)

int32_t CecManagerImpl::OnSystemEvent(nn::sf::Out<uint32_t> pEventFlags,
                                      nn::sf::Out<uint32_t> pCount,
                                      nn::sf::Out<uint64_t> pData0,
                                      nn::sf::Out<uint64_t> pData1) NN_NOEXCEPT
{
    struct EventData    eventData;

    s_TopLevelEventHandler.CopyEventData(&eventData);
    pEventFlags.Set(eventData.busEvent);
    pCount.Set(eventData.count);
    pData0.Set(eventData.eventData0);
    pData1.Set(eventData.eventData1);
    NN_CEC_INFO("%s: flags %d\n", NN_CURRENT_FUNCTION_NAME, eventData.busEvent);
    return CecNoError;
}

}}}
